<?php

namespace Database\Factories;

use App\Models\Asset;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;
use App\Models\RevivnAsset;
use App\Models\RevivnPickup;
use Facades\App\Services\Revivn\RevivnSyncService;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\RevivnAsset>
 */
class RevivnAssetFactory extends Factory
{

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        // Define valid models, hard drives, and CPUs for each manufacturer and asset type
        $models = [
            'Apple' => [
                'Laptop' => [
                    'models' => ['MacBookAir8,2', 'MacBookPro15,2', 'MacBook12,1'],
                    'hard_drives' => ['Apple APPLE SSD AP0128 [' . $this->faker->regexify('[A-Z0-9]{12}') . '] - 121 GB - SCSI ( - GPT)'],
                    'cpus' => ['INTEL(R) CORE(TM) I5-8210Y CPU @ 1.60GHZ', 'INTEL(R) CORE(TM) I7-8557U CPU @ 1.70GHZ']
                ],
                'Desktop' => [
                    'models' => ['iMac19,2', 'Macmini8,1', 'iMacPro1,1'],
                    'hard_drives' => ['Apple APPLE SSD AP0256 [' . $this->faker->regexify('[A-Z0-9]{12}') . '] - 256 GB - SCSI ( - GPT)'],
                    'cpus' => ['INTEL(R) CORE(TM) I5-8500 CPU @ 3.00GHZ', 'INTEL(R) XEON(R) W-2140B CPU @ 3.20GHZ']
                ]
            ],
            'Dell' => [
                'Laptop' => [
                    'models' => ['Latitude 7480', 'XPS 13', 'Inspiron 15'],
                    'hard_drives' => ['512GB', '1TB'],
                    'cpus' => ['INTEL(R) CORE(TM) I5-7200U CPU @ 2.50GHZ', 'INTEL(R) CORE(TM) I7-7700HQ CPU @ 2.80GHZ']
                ],
                'Desktop' => [
                    'models' => ['OptiPlex 7070', 'Inspiron 3671', 'Vostro 3670'],
                    'hard_drives' => ['512GB', '1TB'],
                    'cpus' => ['INTEL(R) CORE(TM) I5-9500 CPU @ 3.00GHZ', 'INTEL(R) CORE(TM) I7-9700 CPU @ 3.00GHZ']
                ]
            ],
            'Xerox' => [
                'Printer' => [
                    'models' => ['WorkCentre 6515', 'VersaLink C400', 'Phaser 6510'],
                    'hard_drives' => [null],
                    'cpus' => [null]
                ]
            ],
        ];

        // Select a random manufacturer
        $manufacturer = $this->faker->randomElement(array_keys($models));

        // Select a random device type based on the manufacturer
        $deviceType = $this->faker->randomElement(array_keys($models[$manufacturer]));

        // Select a random model, hard drive, and CPU based on the manufacturer and device type
        $model = $this->faker->randomElement($models[$manufacturer][$deviceType]['models']);
        $hardDrive = $this->faker->randomElement($models[$manufacturer][$deviceType]['hard_drives']);
        $cpu = $this->faker->randomElement($models[$manufacturer][$deviceType]['cpus']);
        $serialNo = $this->faker->regexify('[A-Z0-9]{8}');
        $action = $this->faker->randomElement(['1', '2', '3', '4', 'update']);

        return [
            'asset_id' => $this->getAssetId($serialNo, $action),
            'serial_no' => $serialNo,
            'asset_tag' => null,
            'revivn_id' => Str::uuid(),
            'revivn_pickup_id' => RevivnPickup::inRandomOrder()->first()->id,
            'model' => $model,
            'device_type' => $deviceType,
            'manufacturer' => $manufacturer,
            'hard_drive' => $hardDrive,
            'cpu' => $cpu,
            'ram' => $this->faker->randomElement(['4GB', '8GB', '16GB', '32GB']),
            'year' => $this->faker->year,
            'hdd' => null,
            'lock_codes' => null,
            'cod_url' => $this->faker->url,
            'end_of_life_condition' => $this->faker->randomElement(['Fair', 'Good', 'Poor']),
            'status' => 'Retired',
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }

    private function getAssetId($serialNo, $action)
    {
        if ($action === 'update') {
            $asset = Asset::hasAssetType(['laptop', 'desktop', 'printer'])->inRandomOrder()->first();
        } else {
            $asset = RevivnSyncService::createAsset($serialNo);
        }
        $ewasteCertificate = 'eol_certificates/revivn/Demo_EOL_Certificate.pdf';
        RevivnSyncService::updateAsset($asset, $ewasteCertificate);
        RevivnSyncService::createAssetHistory($asset);
        return $asset->id;
    }
}
