<?php

namespace Database\Factories;

use App\Models\Asset;
use App\User;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\RetrieverDevice>
 */
class RetrieverDeviceFactory extends Factory
{
    protected static $usedAssetIds = [];
    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $user = User::has('assets')->inRandomOrder()->first();
        $userEmail = $this->faker->unique()->safeEmail;
        $userName = $user ? $user->name : $this->faker->unique()->name;
        $userId = $user ? $user->id : 0;

        // Ensure unique asset_id
        $asset = null;
        if ($user) {
            $availableAssets = Asset::where('user_id', $user->id)->whereNotIn('id', self::$usedAssetIds)->inRandomOrder()->first();
            if ($availableAssets) {
                $asset = $availableAssets;
                self::$usedAssetIds[] = $asset->id; // Track used asset IDs
            }
        }

        $assetId = $asset ? $asset->id : 0;
        $serialNo = $asset ? $asset->serial_no : strtoupper($this->faker->unique()->bothify('??####??'));

        $deactivationReason = null;
        $shipmentStatus = $this->faker->randomElement(array_keys(config('retriever.shipment_statuses')));
        if ($shipmentStatus === 'deactivated') {
            $deactivationReason = $this->faker->text;
        }

        // Generate dates
        $boxShippedAt = $this->faker->dateTimeBetween('2023-01-01', 'now');
        $boxDeliveredAt = (clone $boxShippedAt)->modify('+' . mt_rand(2, 15) . ' days');

        $deviceShippedAt = $this->faker->dateTimeBetween($boxDeliveredAt, (clone $boxDeliveredAt)->modify('+' . mt_rand(2, 15) . ' days'));
        $deviceDeliveredAt = (clone $deviceShippedAt)->modify('+' . mt_rand(2, 15) . ' days');

        return [
            'user_id' => $userId,
            'asset_id' => $assetId,
            'user_email' => $userEmail,
            'user_name' => $userName,
            'city' => $this->faker->city,
            'serial_no' => $serialNo,
            'shipment_status' => $shipmentStatus,
            'deactivated_reason' => $deactivationReason,
            'device_type' => $this->faker->randomElement(array_keys(config('retriever.device_types'))),
            'request_charger' => $this->faker->boolean,
            'outbound_tracking' => $this->faker->bothify('##############'),
            'outbound_carrier' => $this->faker->randomElement(['USPS', 'UPS', 'FedEx']),
            'return_tracking' => $this->faker->bothify('##############'),
            'return_carrier' => $this->faker->randomElement(['USPS', 'UPS', 'FedEx']),
            'box_shipped_at' => $boxShippedAt,
            'box_delivered_at' => $boxDeliveredAt,
            'device_shipped_at' => $deviceShippedAt,
            'device_delivered_at' => $deviceDeliveredAt,
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }
}
