<?php

use Illuminate\Support\Facades\Log;

if (!function_exists('convert_to_db_date')) {
    function convert_to_db_date($date)
    {
        if ($date && DateTime::createFromFormat(config('date.formats.read_date_format'), $date) !== false) {
            return !empty($date) ? Carbon\Carbon::createFromFormat(config('date.formats.read_date_format'), $date)->format('Y-m-d') : null;
        }

        return null;
    }
}

if (!function_exists('convert_to_db_datetime')) {
    function convert_to_db_datetime($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat(config('date.formats.read_date_format'), $date)->format('Y-m-d H:i:s') : null;
    }
}

if (!function_exists('format_date_to_carbon')) {
    function format_date_to_carbon($date)
    {

        if (!empty($date)) {
            return Carbon\Carbon::createFromFormat(config('date.formats.read_date_format'), $date);
        }

        return null;
    }
}

if (!function_exists('format_db_date_time_to_carbon')) {
    function format_db_date_time_to_carbon($date)
    {
        return Carbon\Carbon::createFromFormat("Y-m-d H:i:s", $date);
    }
}

if (!function_exists('format_date_from_year_to_carbon')) {
    function format_date_from_year_to_carbon($year)
    {
        return Carbon\Carbon::createFromFormat("Y", $year);
    }
}

if (!function_exists('parse_date_from_db_date')) {
    function parse_date_from_db_date($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d", $date)->format(config('date.formats.read_date_format')) : null;
    }
}

if (!function_exists('parse_date_from_db_datetime')) {
    function parse_date_from_db_datetime($date)
    {
        if (!empty($date) && DateTime::createFromFormat('Y-m-d H:i:s', $date) !== false) {
            return  Carbon\Carbon::createFromFormat("Y-m-d H:i:s", $date)->format(config('date.formats.read_date_format'));
        }
        return null;
    }
}

if (!function_exists('parse_time_from_db_datetime')) {
    function parse_time_from_db_datetime($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d H:i:s", $date)->format(config('date.formats.read_time_format')) : null;
    }
}

if (!function_exists('parse_datetime_from_db_datetime')) {
    function parse_datetime_from_db_datetime($date)
    {
        if (!empty($date) && DateTime::createFromFormat('Y-m-d H:i:s', $date) !== false) {

            return Carbon\Carbon::createFromFormat("Y-m-d H:i:s", $date)->format(config('date.formats.read_date_time_format'));
        }
        return null;
    }
}

if (!function_exists('format_workspace_one_datetime_to_db_datetime')) {
    function format_workspace_one_datetime_to_db_datetime($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d\TH:i:s.u", $date)->format('Y-m-d H:i:s') : null;
    }
}

if (!function_exists('parse_date_to_month_year')) {
    function parse_date_to_month_year($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat(config('date.formats.read_date_format'), $date)->format(config('date.formats.month_year_format')) : null;
    }
}

if (!function_exists('format_ad_datetime_to_db_datetime')) {

    function format_ad_datetime_to_db_datetime($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("YmdHis.0Z", $date)->format('Y-m-d H:i:s') : null;
    }
}

if (!function_exists('format_ad_datetime_to_db_date')) {
    function format_ad_datetime_to_db_date($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("YmdHis.0Z", $date)->format('Y-m-d') : null;
    }
}


if (!function_exists('parse_date_okta_hire_date')) {
    function parse_date_okta_hire_date($date)
    {
        $formats = ['Y-m-d', 'm/d/Y', 'd/m/Y'];

        foreach ($formats as $format) {
            $dateObject = DateTime::createFromFormat($format, $date);
            // Check if parsing was successful and the formatted date matches the original input to avoid conflicts b/w 'm/d/Y' and 'd/m/Y' format
            if ($dateObject !== false && $dateObject->format($format) === $date) {
                return !empty($date) ? Carbon\Carbon::createFromFormat($format, $date)->format('Y-m-d') : null;
            }
        }
        return null;
    }
}

if (!function_exists('format_ms_timestamp_to_db_datetime')) {
    function format_ms_timestamp_to_db_datetime($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromTimestampMs($date)->format('Y-m-d H:i:s') : null;
    }
}

if (!function_exists('format_fedex_date')) {
    function format_fedex_date($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d\TH:i:s", $date)->format('Y-m-d') : null;
    }
}

if (!function_exists('format_kandji_check_in_date')) {
    function format_kandji_check_in_date($date)
    {
        //202021-04-06 19:45:27.100221+00:00
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d\TH:i:s.uZ", $date)->format('Y-m-d H:i:s') : null;
    }
}

if (!function_exists('format_kandji_date')) {
    function format_kandji_date($date)
    {
        //202021-04-06 19:45:27.100221+00:00
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d H:i:s.uP", $date)->format('Y-m-d H:i:s') : null;
    }
}

/**
 * To convert date from timestamp to desired format.
 *
 * @param int $timestamp
 * @param string $format
 * @return string
 */
if (!function_exists('timestampToDesiredFormat')) {
    function timestampToDesiredFormat($timestamp, $format = 'm/d/Y')
    {
        return Carbon\Carbon::createFromTimestamp($timestamp)->format($format);
    }
}

if (!function_exists('parse_api_date')) {
    function parse_api_date($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d H:i:s", $date)->setTimezone('UTC')->format('Y-m-d\TH:i:s\Z') : null;
    }
}


if (!function_exists('general_parse_datetime')) {
    function general_parse_datetime($date)
    {
        return !empty($date) ? Carbon\Carbon::parse($date)->format('Y-m-d H:i:s') : null;
    }
}


/* Converts db date time to 12 hour format
 * @param mixed $date
 *
 * @return [type]
 */
if (!function_exists('parse_gerneral_datetime')) {
    function parse_gerneral_datetime($date)
    {
        if (DateTime::createFromFormat('Y-m-d H:i:s', $date) !== false) {
            return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d H:i:s", $date)->format(config('date.formats.read_date_time_format_hia')) : null;
        }
        return null;
    }
}

/**
 * Formats a date in ISO date time format to a specific format.
 *
 * @param string $date The date to be formatted.
 * @return string|null The formatted date or null if the date is empty or not in the correct format.
 */
if (!function_exists('format_date_iso_datetime_to_db_datetime')) {
    function format_date_iso_datetime_to_db_datetime($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("Y-m-d\TH:i:sZ", $date)->format('Y-m-d H:i:s') : null;
    }
}

/**
 * Formats a date in ISO date time format to a specific format.
 *
 * @param string $date The date to be formatted.
 * @throws Exception If the date is not in the correct format.
 * @return string|null The formatted date or null if the date is empty or not in the correct format.
 */
if (!function_exists('format_date_iso_datetime_to_db_datetime_v2')) {
    function format_date_iso_datetime_to_db_datetime_v2($date)
    {
        //2023-03-07T23:44:31.666543Z
        if (!empty($date) && DateTime::createFromFormat('Y-m-d\TH:i:s.uZ', $date) !== false) {
            return Carbon\Carbon::createFromFormat("Y-m-d\TH:i:s.uZ", $date)->format('Y-m-d H:i:s');
        }
        return null;
    }
}

if (!function_exists('getMonthName')) {
    function getMonthName($n)
    {  //changed function to private (or protected/public if you need)
        return date('F', mktime(0, 0, 0, $n, 1));
    }
}

if (!function_exists('get_months')) {
    function get_months($start = 1, $end = 12, $count = 1)
    {
        // $start
        // return array_map("getMonthName", range($start, $end));
        $i = 1;
        while ($i <= $count) {
            $dates[$start] = getMonthName($start);
            $start++;
            $i++;
        }
        return $dates;
    }
}

if (!function_exists('get_past_month_names')) {
    function get_past_month_names($dateFromCalc, $dateTo)
    {
        $dates = [];

        while ($dateFromCalc->toDateString() < $dateTo->toDateString()) {
            $dates[$dateFromCalc->format('n')] = $dateFromCalc->format('M Y');
            $dateFromCalc = $dateFromCalc->addMonth();
        }

        return $dates;
    }
}

if (!function_exists('get_past_month_names_day')) {

    function get_past_month_names_day($dateFromCalc, $dateTo)
    {
        $dates = [];

        while ($dateFromCalc->toDateString() <= $dateTo->toDateString()) {
            $dates[$dateFromCalc->format('n')] = $dateFromCalc->format('M Y');
            $dateFromCalc = $dateFromCalc->addDays(28);
        }

        return $dates;
    }
}

if (!function_exists('get_month_year_names')) {
    function get_month_year_names($dateFromCalc, $dateTo)
    {
        $dates = [];

        while ($dateFromCalc->toDateString() < $dateTo->toDateString()) {
            $dates[$dateFromCalc->format('n-Y')] = $dateFromCalc->format('M Y');
            $dateFromCalc = $dateFromCalc->addMonth();
        }

        return $dates;
    }
}

//this function returns 12 months always even if some months are future dates
if (!function_exists('get_month_year_names_with_all_months')) {
    function get_month_year_names_with_all_months($dateFromCalc)
    {

        $dates = [];

        // Get the current month and year
        $currentMonth = $dateFromCalc->format('n');
        $currentYear = $dateFromCalc->format('Y');

        // Add the current month to the dates array
        $dates[$currentMonth . '-' . $currentYear] = $dateFromCalc->format('M Y');

        // Loop through the next 11 months
        for ($i = 0; $i < 11; $i++) {
            // Increment the current month by 1
            $currentMonth++;

            // If the current month exceeds 12, increment the year and reset the month to 1
            if ($currentMonth > 12) {
                $currentMonth = 1;
                $currentYear++;
            }

            // Add the month to the dates array
            $dates[$currentMonth . '-' . $currentYear] = Carbon\Carbon::create($currentYear, $currentMonth, 1)->format('M Y');
        }

        return $dates;
    }
}

if (!function_exists('first_day_current_month_last_year')) {
    function first_day_current_month_last_year()
    {
        return new Carbon\Carbon('first day of ' . Carbon\Carbon::now()->addMonth()->format('F') . ' ' . Carbon\Carbon::now()->subYear()->format('Y'));
    }
}

if (!function_exists('first_day_current_month_future_year')) {
    function first_day_current_month_future_year($age)
    {
        return new Carbon\Carbon('first day of ' . Carbon\Carbon::now()->addMonth()->format('F') . ' ' . Carbon\Carbon::now()->addYears($age)->format('Y'));
    }
}

if (!function_exists('first_day_current_month_past_year')) {
    function first_day_current_month_past_year($age)
    {
        return new Carbon\Carbon('first day of ' . Carbon\Carbon::now()->addMonth()->format('F') . ' ' . Carbon\Carbon::now()->subYears($age)->format('Y'));
    }
}

if (!function_exists('group_by_months')) {
    function group_by_months($date)
    {
        return $date->groupBy(function ($date) {
            return intval(Carbon\Carbon::parse($date->created_at)->format('m')); // grouping by months
        });
    }
}

if (!function_exists('group_by_months_history')) {
    function group_by_months_history($date)
    {
        return $date->groupBy(function ($date) {
            return intval(Carbon\Carbon::parse($date->updated_at)->format('m')); // grouping by months
        });
    }
}


if (!function_exists('get_week_first_day')) {
    function get_week_first_day()
    {
        return Carbon\Carbon::now()->startOfWeek(Carbon\Carbon::SUNDAY);
    }
}

if (!function_exists('get_next_week_last_day')) {
    function get_next_week_last_day()
    {
        return Carbon\Carbon::now()->addWeek()->endOfWeek();
    }
}

if (!function_exists('get_next_second_week_last_day')) {
    function get_next_second_week_last_day()
    {
        return Carbon\Carbon::now()->addWeek(2)->endOfWeek();
    }
}

/**
 * calculate date difference of a date from now
 */
if (!function_exists('calculate_diff_in_days_from_today')) {
    function calculate_diff_in_days_from_today($date)
    {
        $date = Carbon\Carbon::parse($date);
        $now = Carbon\Carbon::now();

        return $date->diffInDays($now);
    }
}

/**
 * check a date is past or not
 */
if (!function_exists('past_date_or_not')) {
    function past_date_or_not($date)
    {
        if (!$date) {
            return false;
        }
        return Carbon\Carbon::createFromFormat(config('date.formats.read_date_format'), $date)->isPast();
    }
}

/**
 * check a date string is valid or not
 */
if (!function_exists('valid_date_or_not')) {
    function valid_date_or_not($date)
    {
        $dateArray = explode('/', $date);

        if ($dateArray && count($dateArray) == 3 && strlen($dateArray[2]) == 4) {
            return checkdate((int)$dateArray[0], (int)$dateArray[1], (int)$dateArray[2]);
        }

        return false;
    }
}

/**
 * add months to the date
 */
if (!function_exists('addMonths')) {
    function addMonths($date, $months)
    {
        return Carbon\Carbon::createFromFormat(config('date.formats.read_date_format'), $date)->addMonths($months)->format(config('date.formats.read_date_format'));
    }
}
/**
 * add years to the date
 */
if (!function_exists('addYears')) {
    function addYears($date, $years)
    {
        return Carbon\Carbon::createFromFormat(config('date.formats.read_date_format'), $date)->addYears($years)->format(config('date.formats.read_date_format'));
    }
}


if (!function_exists('age')) {
    function age($varDate)
    {
        if (Carbon\Carbon::now()->diff($varDate)->format('%y') == 0) {
            if (Carbon\Carbon::now()->diff($varDate)->format('%m') == 0) {
                return Carbon\Carbon::now()->diff($varDate)->format('%d Days');
            } else {
                return Carbon\Carbon::now()->diff($varDate)->format('%y Year %m Months');
            }
        }

        return Carbon\Carbon::now()->diff($varDate)->format('%y Year %m Months %d Days');
    }
}

if (!function_exists('isDateString')) {
    function isDateString($date)
    {
        $date = trim($date);
        if ($date && true == strtotime($date)) {
            return true;
        } else {
            return false;
        }
    }
}


/* Check a date valid or not .
*/
if (!function_exists('validateDate')) {
    function validateDate($date, $format = 'mm/dd/yyyy')
    {
        $date = substr($date, 0, 10);
        if ($date != null) {
            switch ($format) {
                case 'yyyy-mm-dd':
                    $splitDate = explode('-', $date);
                    $m = isset($splitDate[1]) ? (int)$splitDate[1] : 0;
                    $d = isset($splitDate[2]) ? (int)$splitDate[2] : 0;
                    $y = isset($splitDate[0]) ? (int)$splitDate[0] : 0;
                    return checkdate($m, $d, $y);
                default:
                    $splitDate = explode('/', $date);
                    $m = isset($splitDate[0]) ? (int)$splitDate[0] : '';
                    $d = isset($splitDate[1]) ? (int)$splitDate[1] : '';
                    $y = isset($splitDate[2]) ? (int)$splitDate[2] : '';
                    return checkdate($m, $d, $y);
            }
        }
        return false;
    }
}

if (!function_exists('first_day_current_month_year')) {
    function first_day_current_month_year()
    {
        $month = Carbon\Carbon::now()->addMonth()->format('F');
        if ($month == 'January') {
            $year = Carbon\Carbon::now()->format('Y');
        } else {
            $year = Carbon\Carbon::now()->subYear()->format('Y');
        }
        return new Carbon\Carbon('first day of ' . $month . ' ' . $year);
    }
}

if (!function_exists('get_past_month_names_short')) {

    function get_past_month_names_short($dateFromCalc, $dateTo)
    {
        $dates = [];

        while ($dateFromCalc->toDateString() < $dateTo->toDateString()) {
            $dates[$dateFromCalc->format('n')] = $dateFromCalc->format('M');
            $dateFromCalc = $dateFromCalc->addMonth();
        }

        return $dates;
    }
}

/**
 * Gets the first day of the next month of last year ex: Current month June 2023 the returns Jul 2022
 * @return [type]
 */
if (!function_exists('first_day_next_month_last_year')) {
    function first_day_next_month_last_year()
    {
        $month = Carbon\Carbon::now()->addMonth()->format('F');
        if ($month == 'January') {
            $year = Carbon\Carbon::now()->format('Y');
        } else {
            $year = Carbon\Carbon::now()->subYear()->format('Y');
        }
        return new Carbon\Carbon('first day of ' . $month . ' ' . $year);
    }
}

/**
 * Generates Month Year in "Ex:Jan 2023" this format
 * @param $format =
 *
 * @return [type]
 */
if (!function_exists('generateRecentMonthsArray')) {
    function generateRecentMonthsArray($format = 'M Y')
    {
        $months = [];

        $startMonth = Carbon\Carbon::now()->subMonths(11); // Start with the current month and go back 11 months

        for ($i = 0; $i < 12; $i++) {
            $months[] = $startMonth->format($format); // Format the month based on the provided format
            $startMonth->addMonth(); // Move to the next month
        }

        return $months;
    }
}

/**
 * Genrates the last one year months in "example = 07" => "Jul" format
 * @param string $format
 *
 * @return [type]
 */
if (!function_exists('generateRecentYearMonthsArrayWithMonthNumber')) {
    function generateRecentYearMonthsArrayWithMonthNumber($format = 'M')
    {
        $months = [];
        $startMonth = Carbon\Carbon::now()->subMonths(11);

        for ($i = 0; $i <= 11; $i++) {
            $months[ltrim($startMonth->format('m'), '0')] = $startMonth->format('M'); // use ltrim to remove leading zero  as laravel collecton never gives key with leading zero.
            $startMonth->addMonth();
        }
        return $months;
    }
}

if (!function_exists('csv_to_array')) {
    function csv_to_array($filename = '', $delimiter = ',')
    {
        if (!file_exists($filename) || !is_readable($filename)) {
            return false;
        }
        $header = null;
        $data   = array();
        $data['error'] = '';
        $data['csvData'] = array();
        if (($handle = fopen($filename, 'r')) !== false) {
            $i = 0;
            while (($row = fgetcsv($handle, 1000, $delimiter)) !== false) {
                $i++;
                if (!$header) {
                    $header = $row;
                    $data['headers'] = $row;
                } else {
                    if (count($header) == count($row)) {
                        $data['csvData'][] = array_combine($header, $row);
                    }
                    if (count($header) != count($row)) {
                        $data['error'] = 'Please make sure the file headers matching with their values';
                    }
                }
            }
            fclose($handle);
        }
        return $data;
    }
}

if (!function_exists('find_sort_order')) {
    function find_sort_order($name, $length = 2)
    {
        $name = strtolower($name);
        $name = str_replace(' ', '', $name);
        // $length = strlen($name);

        $letterToValues = ['a' => 10, 'b' => 11, 'c' => 12, 'd' => 13, 'e' => 14, 'f' => 15, 'g' => 16, 'h' => 17, 'i' => 18, 'j' => 19, 'k' => 20, 'l' => 21, 'm' => 22, 'n' => 23, 'o' => 24, 'p' => 25, 'q' => 26, 'r' => 27, 's' => 28, 't' => 29, 'u' => 30, 'v' => 31, 'w' => 32, 'x' => 33, 'y' => 34, 'z' => 35, '!' => 36, '#' => 37, '$' => 38, '%' => 39, '&' => 40, '(' => 41, ')' => 42, '*' => 43, '+' => 44, ',' => 45, '-' => 46, '.' => 47, '/' => 48, ':' => 49, ';' => 50, '<' => 51, '=' => 52, '>' => 53, '?' => 54, '@' => 55, '[' => 56, ']' => 57, '^' => 58, '_' => 59, '`' => 60, '{' => 61, '|' => 62, '}' => 63, '~' => 64];

        if (strlen($name) >= 2) {
            for ($i = 0; $i < $length; $i++) {
                $character = $name[$i];
                if (!is_numeric($character)) {
                    $value[] = $letterToValues[$character];
                } else {
                    $value[] = $character;
                }
            }
            return implode('', $value);
        }

        if (is_numeric($name)) {
            return $name;
        }

        if ($name) {
            return $letterToValues[$name];
        }

        return 0;
    }
}

if (!function_exists('num_row')) {
    function num_row($page, $limit)
    {
        if (is_null($page)) {
            $page = 1;
        }
        $num = ($page * $limit) - ($limit - 1);
        return $num;
    }
}

if (!function_exists('stringFormat')) {
    function stringFormat($string)
    {
        return ucwords(str_replace("_", " ", $string));
    }
}

if (!function_exists('isJSON')) {
    function isJSON($string)
    {
        return is_string($string) && is_array(json_decode($string, true)) && (json_last_error() == JSON_ERROR_NONE) ? true : false;
    }
}

if (!function_exists('formatUSPhoneNumbers')) {
    function formatUSPhoneNumbers($phoneNumber)
    {
        return sprintf("(%s) %s-%s", substr($phoneNumber, 0, 3), substr($phoneNumber, 3, 3), substr($phoneNumber, 6));
    }
}

if (!function_exists('formatMexicoPhoneNumbers')) {
    function formatMexicoPhoneNumbers($phoneNumber)
    {
        if (substr($phoneNumber, 0, 2) == "52") {
            return sprintf("%s-%s-%s-%s", substr($phoneNumber, 0, 2), substr($phoneNumber, 2, 3), substr($phoneNumber, 5, 3), substr($phoneNumber, 8));
        }

        return sprintf("52-%s-%s-%s", substr($phoneNumber, 0, 2), substr($phoneNumber, 2, 4), substr($phoneNumber, 6));
    }
}

if (!function_exists('formatBrazilPhoneNumbers')) {
    function formatBrazilPhoneNumbers($phoneNumber)
    {
        return sprintf("%s-%s", substr($phoneNumber, 0, 5), substr($phoneNumber, 5));
    }
}

if (!function_exists('formatIndianPhoneNumbers')) {
    function formatIndianPhoneNumbers($phoneNumber)
    {
        if (substr($phoneNumber, 0, 2) == "91") {
            return sprintf("%s %s %s %s", substr($phoneNumber, 0, 2), substr($phoneNumber, 2, 2), substr($phoneNumber, 4, 4), substr($phoneNumber, 8));
        }

        return sprintf("91 %s %s %s", substr($phoneNumber, 0, 2), substr($phoneNumber, 2, 4), substr($phoneNumber, 6));
    }
}

if (!function_exists('formatNetherLandsPhoneNumbers')) {
    function formatNetherLandsPhoneNumbers($phoneNumber)
    {
        return sprintf("%s %s %s %s", substr($phoneNumber, 0, 2), substr($phoneNumber, 2, 3), substr($phoneNumber, 5, 2), substr($phoneNumber, 7));
    }
}

if (!function_exists('formatUKPhoneNumbers')) {
    function formatUKPhoneNumbers($phoneNumber)
    {
        return sprintf("%s %s", substr($phoneNumber, 0, 5), substr($phoneNumber, 5));
    }
}

if (!function_exists('formatCanadaPhoneNumbers')) {
    function formatCanadaPhoneNumbers($phoneNumber)
    {
        return sprintf("(%s) %s-%s", substr($phoneNumber, 0, 3), substr($phoneNumber, 3, 3), substr($phoneNumber, 6));
    }
}

if (!function_exists('isNewFile')) {
    function isNewFile($fileName)
    {
        $fileArray = explode('_', $fileName);
        if ($fileArray[0] == Carbon\Carbon::now()->format("Ymd") || $fileArray[0] == Carbon\Carbon::now()->subDay()->format("Ymd")) {
            return true;
        }

        return false;
    }
}

if (!function_exists('formatSingaporePhoneNumbers')) {
    function formatSingaporePhoneNumbers($phoneNumber)
    {
        if (substr($phoneNumber, 0, 4) == "(65)") {
            return sprintf("%s %s %s", substr($phoneNumber, 0, 4), substr($phoneNumber, 4, 4), substr($phoneNumber, 8));
        }

        if (substr($phoneNumber, 0, 2) == "65") {
            return sprintf("(65) %s %s", substr($phoneNumber, 2, 4), substr($phoneNumber, 6));
        }

        return sprintf("(65) %s %s", substr($phoneNumber, 0, 4), substr($phoneNumber, 4));
    }
}

if (!function_exists('formatFrancePhoneNumbers')) {
    function formatFrancePhoneNumbers($phoneNumber)
    {
        return sprintf("%s %s %s %s %s", substr($phoneNumber, 0, 2), substr($phoneNumber, 2, 2), substr($phoneNumber, 4, 2), substr($phoneNumber, 6, 2), substr($phoneNumber, 8, 2));
    }
}

if (!function_exists('formatAustraliaPhoneNumbers')) {
    function formatAustraliaPhoneNumbers($phoneNumber)
    {
        return sprintf("(%s) %s %s", substr($phoneNumber, 0, 2), substr($phoneNumber, 2, 4), substr($phoneNumber, 6));
    }
}

if (!function_exists('disableCSVInjection')) {
    function disableCSVInjection($string)
    {
        $string = strval($string);
        if ($string) {
            if (in_array($string[0], ['+', '-', '=', '@', '('])) {
                $string = "'" . $string;
            }
        }

        return $string;
    }
}

if (!function_exists('getCanBeUpdatedToStatus')) {
    function getCanBeUpdatedToStatus($status)
    {
        $array = config('asset-status');
        $return = array();
        foreach ($array as $key => $val) {
            if (in_array($status, $val)) {
                $return[] = $key;
            }
        }

        return $return;
    }
}


if (!function_exists('get_csv_header')) {
    function get_csv_header($filename = '', $delimiter = ',')
    {
        if (!file_exists($filename) || !is_readable($filename)) {
            return false;
        }

        $header = null;
        if (($handle = fopen($filename, 'r')) !== false) {
            $i = 0;
            while (($row = fgetcsv($handle, 1000, $delimiter)) !== false) {
                $i++;
                if (!$header) {
                    $header = $row;
                    break;
                }
            }
            fclose($handle);
        }
        return $header;
    }
}

if (!function_exists('excel_to_array')) {
    function excel_to_array($filename = '')
    {
        if (!file_exists($filename) || !is_readable($filename)) {
            return false;
        }
        $header = null;
        $data = array();
        $data['error'] = '';
        $data['csvData'] = array();

        $reader = new \PhpOffice\PhpSpreadsheet\Reader\Xlsx();

        $spreadsheet = $reader->load($filename);

        $worksheet = $spreadsheet->getActiveSheet();

        $worksheetData = $worksheet->toArray();


        $header = [];
        foreach ($worksheetData as $row => $rowData) {
            if ($row == 0) {
                $header = $rowData;
            } elseif (array_filter($rowData)) {
                if (count($header) == count($rowData)) {
                    $rowData = array_map('strval', $rowData);
                    $data['csvData'][] = array_combine($header, $rowData);
                }
                if (count($header) != count($rowData)) {
                    $data['error'] = 'Please make sure the file headers matching with their values';
                }
            }
        }

        return $data;
    }
}

if (!function_exists('get_excel_header')) {
    function get_excel_header($filename = '')
    {
        if (!file_exists($filename) || !is_readable($filename)) {
            return [];
        }

        $header = [];


        $reader = new \PhpOffice\PhpSpreadsheet\Reader\Xlsx();

        $spreadsheet = $reader->load($filename);
        $spreadsheet = $reader->load($filename);

        $worksheet = $spreadsheet->getActiveSheet();

        $worksheetData = $worksheet->toArray();

        foreach ($worksheetData as $row => $rowData) {
            if ($row == 0) {
                $header = $rowData;
            }
        }
        $header = array_filter($header);

        return $header;
    }
}

if (!function_exists('exportToCsv')) {
    function exportToCsv($datas, $csvHeaders = [])
    {
        $headers = array(
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=file.csv",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        );

        $callback = function () use ($datas, $csvHeaders) {
            $file = fopen('php://output', 'w');
            if (isset($datas[0]) && !empty($datas[0])) {
                fputcsv($file, array_keys($datas[0][0]));

                foreach ($datas as $data) {
                    foreach ($data as $data) {
                        fputcsv($file, $data);
                    }
                }
            } else {
                fputcsv($file, array_values($csvHeaders));
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}


/*
* Check the given tracking number is UPS or not
*/
if (!function_exists('isUpsTrackingNo')) {

    function isUpsTrackingNo($trackingNo)
    {
        if ($trackingNo) {
            if (substr($trackingNo, 0, 2) == '1Z') {
                return true;
            }
        }
        return false;
    }
}



// convert technical specs to readable value
if (!function_exists('convertSpecsToReadableValue')) {
    function convertSpecsToReadableValue($size, $techSpec)
    {
        $fileSize = '-';
        switch ($techSpec) {
            case 'processor_speed':
                if (is_numeric($size)) {
                    $fileSize = round($size / 1024, 1) . 'Ghz';
                }
                break;
            case 'ram':
                if (is_numeric($size)) {
                    $fileSize = round($size / 1024);
                    $fileSize = getStandardValue($fileSize);
                    $fileSize = $fileSize . 'GB';
                }
                break;
            case 'disk_capacity':
                if (is_numeric($size)) {
                    $fileSize = round($size / 1024);
                    $fileSize = getStandardValue($fileSize);
                    $fileSize = $fileSize . 'GB';
                }
                break;
        }

        return $fileSize;
    }
}

// convert memory into proper standard value, 1, 2, 4 etc
if (!function_exists('getStandardValue')) {
    function getStandardValue($fileSize)
    {
        $standardValue = 1;
        $flag = 0;

        for ($i = 1; $flag == 0; ++$i) {
            if ($fileSize == $standardValue) {
                $flag = 1;

                return $fileSize;
            } elseif ($fileSize < $standardValue) {
                $flag = 1;
                $fileSize = $standardValue;

                return $standardValue;
            } else {
                $standardValue = $standardValue * 2;
            }
        }

        return $fileSize;
    }
}


if (!function_exists('formatMacAddress')) {
    function formatMacAddress($macAddress)
    {
        if (!$macAddress) {
            return null;
        }

        return strtoupper(implode(':', str_split(str_replace([':', '-', ' '], '', $macAddress), 2)));
    }
}

/**
 * Convert kilohertz to gigahertz
 */
if (!function_exists('convertKilohertzToGigahertz')) {
    function convertKilohertzToGigahertz($khz)
    {
        if ($khz) {
            //The frequency f in hertz (Hz) is equal to the frequency f in kilohertz (kHz) divided by 1000000:
            return $khz / 1000000;
        }

        return $khz;
    }
}

/**
 * convert technical specs to readable value from bytes
 *
 * @param string $size     size in bytes
 * @param string $techspec tech spec type
 */

if (!function_exists('convertSpecsToReadableValueFromBytes')) {
    function convertSpecsToReadableValueFromBytes($size, $techSpec)
    {
        $fileSize = '-';
        switch ($techSpec) {
            case 'processor_speed':
                if (is_numeric($size)) {
                    $fileSize = round(((($size / 1024) / 1024) / 1024), 1) . 'Ghz';
                }
                break;
            case 'ram':
                if (is_numeric($size)) {
                    $fileSize = floor(((($size / 1024) / 1024) / 1024));
                    $fileSize = getStandardValue($fileSize);
                    $fileSize = $fileSize . 'GB';
                }
                break;
            case 'disk_capacity':
                if (is_numeric($size)) {
                    $fileSize = floor(((($size / 1024) / 1024) / 1024));
                    $fileSize = getStandardValue($fileSize);
                    $fileSize = $fileSize . 'GB';
                }
                break;
        }

        return $fileSize;
    }
}

if (!function_exists('getStandardHDDCapacity')) {
    function getStandardHDDCapacity($sizeInGb)
    {
        if (empty($sizeInGb)) {
            return null;
        }

        switch ($sizeInGb) {
            case ($sizeInGb == 16 || $sizeInGb < 16):
                return '16GB';
                break;
            case ($sizeInGb == 32 || $sizeInGb < 32):
                return '32GB';
                break;
            case ($sizeInGb == 64 || $sizeInGb < 64):
                return '64GB';
                break;
            case ($sizeInGb == 128 || $sizeInGb < 128):
                return '128GB';
                break;
            case ($sizeInGb == 256 || $sizeInGb < 256):
                return '256GB';
                break;
            case ($sizeInGb == 500 || $sizeInGb < 500):
                return '500GB';
                break;
            case ($sizeInGb == 1000 || $sizeInGb < 1000):
                return '1TB';
                break;
            case ($sizeInGb == 1500 || $sizeInGb < 1500):
                return '1.5TB';
                break;
            case ($sizeInGb == 2000 || $sizeInGb < 2000):
                return '2TB';
                break;
            case ($sizeInGb == 4000 || $sizeInGb < 4000):
                return '4TB';
                break;
            case ($sizeInGb == 6000 || $sizeInGb < 6000):
                return '6TB';
                break;
            case ($sizeInGb == 8000 || $sizeInGb < 8000):
                return '8TB';
                break;
            case ($sizeInGb == 12000 || $sizeInGb < 12000):
                return '12TB';
                break;
            default:
                return null;
        }
    }
}

/**
 * Log custom messages
 */
if (!function_exists('logMessage')) {
    function logMessage($message, $status)
    {
        if ($status == 'success') {
            Log::info($message);
        } else {
            Log::error($message);
        }
    }
}

/**
 * validatePONumber
 *
 * @param  mixed $poNumber
 * @return void
 */
if (!function_exists('validatePONumber')) {
    function validatePONumber($poNumber)
    {
        if (is_numeric($poNumber)) {
            if ($poNumber < 0) {
                return $poNumber * (-1);
            }
            return (int)$poNumber;
        }
        $po = explode(".", $poNumber);
        return $po[0];
    }
}

/**
 * getLatestFileNames in the folder
 *
 * @param  mixed $path
 * @return void
 */
if (!function_exists('getLatestFileNames')) {
    function getLatestFileNames($path)
    {
        if (!is_dir($path)) {
            return false;
        }

        $files = scandir($path, SCANDIR_SORT_DESCENDING);

        if (!$files) {
            return false;
        }

        return $files;
    }
}
/**
 * getLatestFileName
 *
 * @param  mixed $path
 * @return void
 */
if (!function_exists('getLatestFileName')) {
    function getLatestFileName($path)
    {
        $files = scandir($path, SCANDIR_SORT_DESCENDING);
        if (!$files) {
            return false;
        }

        return $files[0];
    }
}

/**
 * getItems - get rows from uploaded files.
 *
 * @param  mixed $path
 * @return void
 */
if (!function_exists('getItems')) {
    function getItems($path)
    {
        if (!file_exists($path)) {
            return false;
        }
        $extension = getFileExtension($path);
        if ($extension == 'csv') {
            $items = csv_to_array($path);
        } elseif ($extension == 'xlsx') {
            $items = excel_to_array($path);
        }
        if (empty($items)) {
            return false;
        }

        return $items;
    }
}

/**
 * stripS
 *
 * @param  mixed $str
 * @return void
 */
if (!function_exists('stripS')) {
    function stripS($str = '')
    {
        if ($str == '') {
            return $str;
        }
        if ($str[0] == 's' || $str[0] == 'S') {
            $str = substr($str, 1);
        }

        return $str;
    }
}

/**
 *  adding / to the end of the URL
 */
if (!function_exists('add_slash_at_end')) {
    function add_slash_at_end($url)
    {
        if (substr($url, -1) == "/") {
            return $url;
        }

        return $url . "/";
    }
}


if (!function_exists('get_next_week_first_day')) {
    function get_next_week_first_day()
    {
        return Carbon\Carbon::now()->addWeek()->startOfWeek();
    }
}

if (!function_exists('numberFormat')) {
    function numberFormat($number)
    {
        if (is_float($number)) {
            return number_format($number, 2, '.', ',');
        }
        return number_format($number);
    }
}

/**
 * @param mixed $comment
 *
 * @return string
 */
if (!function_exists('formatTicketComment')) {
    function formatTicketComment($comment): string
    {
        if ($comment == '') {
            return '';
        }

        $ticketInegrationClassName = config('ticket-integration.service', null);

        $lineBreak = config('ticket-integration.' . $ticketInegrationClassName . '.newline_character') ?? '/n';

        $boldTextStart = config('ticket-integration.' . $ticketInegrationClassName . '.bold_text_start') ?? '*';
        $boldTextEnd = config('ticket-integration.' . $ticketInegrationClassName . '.bold_text_end') ?? '*';

        return $boldTextStart . "Comments:" . $boldTextEnd . $lineBreak . $comment;
    }
}

/**
 * Calculates the difference in days between a given date and today.
 *
 * @param string $date The date to calculate the difference from. Format: Y-m-d
 *
 * @return int The number of days between the given date and today.
 */
if (!function_exists('get_day_diff_from_today')) {
    function get_day_diff_from_today($date)
    {
        if (empty($date)) {
            return null;
        }

        $dateToSubtract = Carbon\Carbon::parse($date);
        $now = Carbon\Carbon::now();

        return $dateToSubtract->diffInDays($now);
    }
}

if (!function_exists('get_week_range_count')) {
    function get_week_range_count($dateFrom, $dateTo)
    {
        $dates = [];

        while ($dateFrom->toDateString() <= $dateTo->toDateString()) {
            $date = new DateTime($dateFrom);
            $week = $date->format("W");
            $formattedFrom = $dateFrom->format(config('date.formats.read_date_format'));
            $formattedTo = $dateFrom->addDays(6)->format(config('date.formats.read_date_format'));
            $dates[(int)$week] = $formattedFrom;

            $dateFrom = $dateFrom->addDays(1);
        }

        return $dates;
    }
}

if (!function_exists('getDaysDifference')) {
    function getDaysDifference($varDate, $endDate = null)
    {
        if ($varDate) {
            $varDate = format_db_date_time_to_carbon($varDate);
            if ($endDate == null) {
                $endDate = Carbon\Carbon::now();
            }
            return $varDate->diffInDays($endDate) . ' Days';
        }
    }
}

if (!function_exists('create_date_with_year_month_date')) {
    function create_date_with_year_month_date($year, $month = 1, $date = 1)
    {
        return \Carbon\Carbon::createFromDate($year, $month, $date);
    }
}

/**
 * Returns the date that is thirty days before the current date.
 *
 * @return string The formatted date (mm/dd/yyyy).
 */
if (!function_exists('thirty_days_before_current_date')) {
    function thirty_days_before_current_date()
    {
        return Carbon\Carbon::now()->subDays(30)->format(config('date.formats.read_date_format'));
    }
}

/**
 * Returns age of asset when its lost/stolen
 * @param date $assetCreatedDate
 * @param date $date
 *
 * @return string The formatted date (mm/dd/yyyy).
 */
if (!function_exists('calculateAgeToDate')) {
    function calculateAgeToDate($assetCreatedDate, $date)
    {
        $date = Carbon\Carbon::parse($date);
        if ($date->diff($assetCreatedDate)->format('%y') == 0) {
            if ($date->diff($assetCreatedDate)->format('%m') == 0) {
                return $date->diff($assetCreatedDate)->format('%d Days');
            } else {
                return $date->diff($assetCreatedDate)->format('%y Year %m Months');
            }
        }

        return $date->diff($assetCreatedDate)->format('%y Year %m Months %d Days');
    }
}

/**
 * Generates a user link based on the provided user ID and display name.
 *
 * @param string $userId The ID of the user.
 * @param string $displayName The display name of the user.
 * @return string The generated user link.
 */
if (!function_exists('generateUserLink')) {
    function generateUserLink($userId, $displayName)
    {
        if ($userId == '') {
            return $displayName;
        }

        return "<a href=" . route('users.show', $userId) . ">" . $displayName . "</a>";
    }
}

/**
 * Generates an asset link based on the provided asset ID and display name.
 *
 * @param string $assetId The ID of the asset.
 * @param string $displayName The display name of the asset.
 * @return string The generated asset link.
 */
if (!function_exists('generateAssetLink')) {
    function generateAssetLink($assetId, $displayName)
    {
        if ($assetId == '') {
            return $displayName;
        }

        return "<a href=" . route('assets.show', $assetId) . ">" . $displayName . "</a>";
    }
}

/**
 * Generates a location link based on the provided location ID and display name.
 *
 * @param string $locationId The ID of the location.
 * @param string $displayName The display name of the location.
 * @return string The generated location link.
 */
if (!function_exists('generateLocationLink')) {
    function generateLocationLink($locationId, $displayName)
    {
        if ($locationId == '') {
            return $displayName;
        }

        return "<a href=" . url('location', $locationId) . ">" . $displayName . "</a>";
    }
}


/**
 * Calculates the date based on the asset age and age unit.
 *
 * @param int $assetAge The age of the asset.
 * @param string $ageUnit (Optional) The unit of age (day, month, year). Default is empty.
 * @return Carbon\Carbon The calculated date.
 */
if (!function_exists('calculateRelativeDateForAssetAge')) {
    function calculateRelativeDateForAssetAge($assetAge, $ageUnit = '')
    {
        if ($ageUnit == 'day') {
            $date = Carbon\Carbon::now()->subDays($assetAge);
        }

        if ($ageUnit == 'month') {
            $date = Carbon\Carbon::now()->subMonths($assetAge);
        }

        if ($ageUnit == 'year') {
            $date = Carbon\Carbon::now()->subYears($assetAge);
        }

        return $date;
    }
}

/**
 * Parses a cron date and time string into a formatted date and time string.
 *
 * @param string $date The cron date and time string in the format "m/d/Y H:i:s".
 * @return string|null The formatted date and time string in the format "Y-m-d H:i:s", or null if the input is empty.
 */
if (!function_exists('parse_cron_date_time')) {
    function parse_cron_date_time($date)
    {
        return !empty($date) ? Carbon\Carbon::createFromFormat("m/d/Y H:i:s", $date)->format('Y-m-d H:i:s') : null;
    }
}

/**
 * Generates an array of week ranges and their corresponding dates between two given dates.
 *
 * @param string $dateFrom The start date in the format "Y-m-d".
 * @param string $dateTo The end date in the format "Y-m-d".
 * @return array An associative array where the keys are week numbers and the values are dates in the format "m/d/y".
 */
if (!function_exists('get_asset_health_week_range_count')) {
    function get_asset_health_week_range_count($dateFrom, $dateTo)
    {
        $dates = [];

        $currentDate = Carbon\Carbon::parse($dateFrom);
        $endDate = Carbon\Carbon::parse($dateTo);

        while ($currentDate->lessThanOrEqualTo($endDate)) {
            $week = $currentDate->format("W");
            $dates[(int)$week] = $currentDate->format('m/d/y');

            $currentDate->addDay();
        }

        return $dates;
    }
}

if (!function_exists('setUnlimitedExecutionTimeAndMemoryLimit')) {
    function setUnlimitedExecutionTimeAndMemoryLimit()
    {
        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '-1');
    }
}

/**
 * Retrieves the IP address of the client making the request.
 *
 * @return string The IP address of the client making the request.
 */
function getIp()
{
    $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? request()->ip();
    $ipArray = explode(',', $ip);

    return $ipArray[0];
}

/**
 * Parses a HiBob date string into a standardized database datetime format.
 *
 * This function handles two possible HiBob date formats:
 * 1. Full ISO-8601 format with microseconds (Y-m-d\TH:i:s.u)
 * 2. Simple date format (Y-m-d)
 *
 * @param string|null $date The date string to parse from HiBob (can be null or empty)
 * @return string|null Formatted date in 'Y-m-d H:i:s' format, or null if input is invalid
 */
function parse_hibob_date($date)
{
    // Return null for empty input
    if (empty($date)) {
        return null;
    }

    // Try to parse as full ISO-8601 format with microseconds
    if (DateTime::createFromFormat('Y-m-d\TH:i:s.u', $date) !== false) {
        return Carbon\Carbon::createFromFormat('Y-m-d\TH:i:s.u', $date)->format('Y-m-d H:i:s');
    }

    // Try to parse as simple date format (without time)
    if (DateTime::createFromFormat('Y-m-d', $date) !== false) {
        // Append default time for date-only entries
        return Carbon\Carbon::createFromFormat('Y-m-d', $date)->format('Y-m-d 00:00:00');
    }

    // Return null if the input doesn't match any expected format
    return null;
}

/**
 * Returns a string containing a link to the Generated Reports page.
 *
 * @param string $reportType The type of report to generate.
 *
 * @return string A string containing a link to the Generated Reports page.
 */
if (!function_exists('getGeneratedReportLink')) {
    function getGeneratedReportLink($reportType = '')
    {
        return "Your report will be generated in the background. Go to the <a target='_blank' href='/generated-exports/$reportType'>Generated Reports Page</a> to download the report.";
    }
}

if (!function_exists('getFileExtension')) {
    /**
     * Gets the file extension of a given filename.
     *
     * @param string $filename The filename to get the extension from.
     * @return string The file extension.
     */
    function getFileExtension($filename)
    {
        return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    }
}

if (!function_exists('sanitizeFilename')) {
    /**
     * Sanitizes a file path to prevent directory traversal attempts.
     *
     * Removes any directory information from the path and returns only the
     * filename. This is useful when creating a filename from user input.
     *
     * @param string $path The potentially malicious path
     * @return string The sanitized filename
     */
    function sanitizeFilename($path)
    {
        // Remove any directory traversal attempts and return only the basename
        return basename($path);
    }
}

if (!function_exists('sanitizeDirectoryFilename')) {
    /**
     * Sanitizes a filename to prevent directory traversal attempts.
     *
     * Removes any directory traversal attempts from the filename and returns the
     * sanitized filename. This is useful when creating a filename from user input.
     *
     * @param string $file The potentially malicious filename
     * @return string The sanitized filename
     */
    function sanitizeDirectoryFilename($file)
    {
        // Remove any ../ or ..\ to prevent directory traversal
        $file = str_replace(['../', '..\\'], '', $file);

        // Optional: remove leading slashes
        return ltrim($file, '/\\');
    }
}