<?php

namespace App;


use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;
use App\Models\UserType;
use Spatie\Permission\Traits\HasRoles;
use App\Notifications\NewPost;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Laravel\Passport\HasApiTokens;
use App\Events\UserCreated;

class User extends Authenticatable
{
    use Notifiable, HasRoles, HasFactory, HasApiTokens;

    protected $guarded = [];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'remember_token',
    ];

    /**
     * The event map for the model.
     *
     * @var array
     */
    protected $dispatchesEvents = [
        'created' => UserCreated::class,
    ];

    public function scopePeople($query)
    {
        return $query->whereIn('user_type_id', [
            UserType::people()->pluck('id')
        ]);
    }

    public function scopePeoples($query)
    {
        return $query->whereIn('user_type_id', [
            UserType::peoples()->pluck('id')
        ]);
    }

    public function scopeAdmin($query)
    {
        return $query->whereIn(
            'user_type_id',
            UserType::admin()->pluck('id')
        )->orderBy('first_name', 'asc');
    }

    public function scopeAdminUsers($query)
    {
        return $query->whereIn(
            'user_type_id',
            UserType::adminUsers()->pluck('id')
        )->orderBy('first_name', 'asc');
    }

    public function scopeRoles($query)
    {
        return $query->whereIn(
            'user_type_id',
            UserType::roles()->pluck('id')
        )->orderBy('first_name', 'asc');
    }

    public function scopeUserRoles($query)
    {
        return $query->whereIn(
            'user_type_id',
            UserType::userRoles()->pluck('id')
        );
    }

    public function scopeSuperAdmin($query)
    {
        return $query->where('user_type_id', '!=', 5);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeInActive($query)
    {
        return $query->whereNull('status')->orWhere('status', '!=', 1);
    }

    public function employeeType()
    {
        return $this->belongsTo('App\Models\EmployeeType');
    }

    public function userType()
    {
        return $this->belongsTo('App\Models\UserType');
    }

    public function department()
    {
        return $this->belongsTo('App\Models\Department');
    }

    public function nuber()
    {
        return $this->belongsTo('App\Models\Nuber');
    }

    public function location()
    {
        return $this->belongsTo('App\Models\Location');
    }

    public function country()
    {
        return $this->belongsTo('App\Models\Country');
    }

    public function position()
    {
        return $this->belongsTo('App\Models\EmployeePosition');
    }

    public function assets()
    {
        return $this->hasMany('App\Models\Asset');
    }

    public function mobile_phone()
    {
        return $this->assets()->mobileAsset();
    }

    public function network()
    {
        return $this->assets()->networkAsset();
    }

    public function av()
    {
        return $this->assets()->avAsset();
    }

    public function manager()
    {
        return $this->hasOne('App\User', 'id', 'manager_id');
    }

    public function haveManager()
    {
        return $this->belongsTo('App\User', 'manager_id');
    }

    public function research()
    {
        return $this->assets()->researchAsset();
    }

    public function getHireDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setChangeEffectiveDateAttribute($date)
    {
        return $this->attributes['change_effective_date'] =  convert_to_db_date($date);
    }

    public function getChangeEffectiveDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    // public function setTerminatedDateAttribute($date)
    // {
    //     return $this->attributes['terminated_date'] =  convert_to_db_date($date);
    // }

    public function getTerminatedDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setOriginalHireDateAttribute($date)
    {
        return $this->attributes['original_hire_date'] =  convert_to_db_date($date);
    }

    public function getOriginalHireDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function scopeTestMail()
    {
        $post = 'mail';
        // NewPost::send(User::find(1), new NewPost());
        User::find(1)->notify(new NewPost());
    }

    public function getUserNameAttribute()
    {
        if (!empty($this->attributes['first_name']) || !empty($this->attributes['last_name'])) {
            return $this->attributes['first_name'] . ' ' . $this->attributes['last_name'];
        }

        return $this->attributes['email'] ?? '';
    }

    public function scopeUserWhere($query, $searchText)
    {
        return $query->where('email', 'like', '%' . $searchText . '%');
    }

    public function scopeUserHasMobileAssignedOrLoaned($query, $userId)
    {
        return $query->whereHas('assets', function ($query) {
            $query->hasAssetType('mobile_phone')->hasAnyAssetStatus(['assigned', 'loaned']);
        })->where('id', $userId)->count();
    }

    public function scopeUserHasLaptopAssignedOrLoaned($query, $userId)
    {
        return $query->whereHas('assets', function ($query) {
            $query->hasAssetType('laptop')->hasAnyAssetStatus(['assigned', 'loaned']);
        })->where('id', $userId)->count();
    }

    public function scopeUserHasAssetAssignedOrLoaned($query, $userId)
    {
        return $query->whereHas('assets', function ($query) {
            $query->hasAssetType(['mobile_phone', 'laptop'])->hasAnyAssetStatus(['assigned', 'loaned']);
        })->where('id', $userId)->count();
    }


    public function scopeUserWhereEmail($query, $searchText)
    {
        return $query->where('email', 'like', $searchText);
    }

    public function scopeLegalHold($query)
    {
        return $query->where('legalhold', 1);
    }

    public function scopeNotLegalHold($query)
    {
        return $query->whereNull('legalhold')->orWhere('legalhold', '!=', 1);
    }

    public function scopeHasMobilePhone($query)
    {
        return $query->whereHas('assets', function ($query) {
            $query->isMobile();
        });
    }

    public function scopeHasNoMobilePhone($query)
    {
        return $query->wheredoesntHave('assets', function ($query) {
            $query->isMobile();
        });
    }

    public function scopeHasMobilePhoneAssigned($query)
    {
        return $query->whereHas('assets', function ($query) {
            $query->isMobile()->hasStatus('assigned');
        });
    }

    public function scopeUserHasMobilePhoneAssigned($query, $userId)
    {
        return $query->whereHas('assets', function ($query) {
            $query->isMobile()->hasStatus('assigned');
        })->where('id', $userId)->count();
    }

    public function userJiraMappings()
    {
        return $this->hasOne('App\Models\UserJiraMappings');
    }


    public function jamfComputer()
    {
        return $this->belongsTo('App\Models\JamfComputer', 'id', 'user_id');
    }

    /**
     * Relation with User and assigned licenses for the user
     */
    public function userLicense()
    {
        return $this->hasMany('App\Models\SoftwareLicense\SoftwareLicenseUser');
    }
    //user having term assets
    public function termAsset()
    {
        return $this->assets()->termAsset();
    }

    //user having term assetscomments
    public function termAssetsComment()
    {
        return $this->hasMany('App\Models\TermAssetsComment');
    }

    /**
     * Finds the super admins
     * @param mixed $query
     *
     * @return [type]
     */
    public function scopeTeqtivityUser($query)
    {
        return $query->where('user_type_id', UserType::where('name', 'Super Admin')->first()->id);
    }

    /**
     * Send a custom password reset notification to the user.
     *
     * @param  string  $token
     * @return void
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new \App\Notifications\ResetPassword($token));
    }

    //user having term assetscomments
    public function favouriteReports()
    {
        return $this->hasMany('App\Models\FavouriteReport');
    }

    /**
     * filter the users having hire date between 2 days
     */
    public function scopeWhereBetweenHireDate($query, $fromDate, $toDate)
    {
        return $query->whereBetween('hire_date', [$fromDate, $toDate]);
    }

    /**
     * filter the users having terminated date between 2 days
     */
    public function scopeWhereBetweenTerminatedDate($query, $fromDate, $toDate)
    {
        return $query->whereBetween('terminated_date', [$fromDate, $toDate]);
    }


    /**
     * Count of destop or laptop assigned
     */
    public function scopecountDesktopOrLaptopAssigned($query)
    {
        return $query->withCount([
            'assets' => function ($query) {
                $query->hasAnyAssetType(['laptop', 'desktop'])->hasStatus('assigned');
            }
        ]);
    }

    public function newUserAssetHistory()
    {
        return $this->belongsTo('App\Models\AssetHistory', 'id', 'new_user_id');
    }

    /**
     * A description of the entire PHP function.
     *
     * @param datatype $paramname description
     * @throws Some_Exception_Class description of exception
     * @return Some_Return_Value
     */
    public function receivedAsset()
    {
        return $this->assets()->receivedAsset();
    }
}
