<?php

namespace App\Services\Zoom;

use App\Events\BulkUpdates;
use App\Http\Traits\AutoCreateInstallTrait;
use App\Models\Country;
use App\Models\Location;
use App\Models\LocationType;
use App\Models\SiteCode;
use App\Models\ZoomDevice;
use App\Repositories\DiscoveryTools\ZoomRepository;
use App\Services\Integrations\ZoomHardwareIntegration;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ZoomHardwareSyncService
{
    use AutoCreateInstallTrait;

    /**
     * Constructs a new instance of the ZoomSyncService class.
     *
     * @param ZoomHardwareCredentialsService $credentialsService The ZoomHardwareCredentialsService instance used for API authentication.
     * @param ZoomHardwareIntegration $apiService The ZoomHardwareIntegration instance used for API integration.
     * @param ZoomRepository $zoomRepository The ZoomRepository instance used for database operations.
     */
    public function __construct(
        protected ZoomHardwareCredentialsService $credentialsService,
        protected ZoomHardwareIntegration $apiService,
        protected ZoomRepository $zoomRepository
    ) {}

    /**
     * Retrieves all Zoom rooms and its device details using the Zoom API and saves the data in the database.
     *
     * @param bool $download If true, returns only the response data for downloading.
     *
     * @return array List of Zoom rooms
     */
    public function syncZoomRooms($download = false)
    {
        $rooms         = [];
        $nextPageToken = '';
        $count         = 0;

        try {
            $allZoomLocations = $this->getAllZoomRoomLocations();

            do {
                $accessToken = $this->credentialsService->getCurrentAccessToken();
                $query = [
                    'type' => 'ZoomRoom',
                    'next_page_token' => $nextPageToken,
                ];
                $response      = $this->apiService->getApiResponse($accessToken, 'Zoom Rooms', '/rooms', $query);
                $nextPageToken = $response['next_page_token'] ?? '';
                $responseData  = $response['rooms'] ?? [];

                // Get the response data for downloading.
                if ($download) {
                    $rooms = array_merge($rooms, $responseData);
                    continue;
                }

                foreach ($responseData as $room) {
                    $roomData = $this->extractRoomDetails($room, $allZoomLocations);

                    sleep(5);   // Wait for 5 seconds before fetching the next room devices.

                    $devices = $this->getZoomRoomDevices($roomData['room_id']);

                    if (empty($devices)) {
                        continue;
                    }

                    foreach ($devices as $device) {
                        $dataToUpdate = $this->parseZoomData($roomData, $device);
                        $this->zoomRepository->updateOrCreateZoomData($dataToUpdate);

                        if (!$dataToUpdate['asset_id'] && $dataToUpdate['location_id'] > 0) {
                            $this->autoCreateInstall($dataToUpdate, 'Zoom');
                        }
                    }

                    $count += count($devices);

                    echo "\n$count Zoom devices has been synced.\n";
                }
            } while (!empty($nextPageToken));

            // If download is true, return the response data.
            if ($download) {
                return $rooms;
            }

            return true;
        } catch (Exception $e) {
            Log::error('Zoom API sync error: ' . $e->getMessage() . $e->getTraceAsString());

            return false;
        }
    }

    /**
     * Retrieves all Zoom room locations using the Zoom API.
     *
     * @return array List of Zoom room locations
     */
    public function getAllZoomRoomLocations()
    {
        $locations = [];
        $nextPageToken = '';

        do {
            $accessToken = $this->credentialsService->getCurrentAccessToken();

            $query = [
                'next_page_token' => $nextPageToken,
            ];

            $response = $this->apiService->getApiResponse($accessToken, 'Zoom Locations', '/rooms/locations', $query);

            $locations = array_merge($locations, $response['locations'] ?? []);
            $nextPageToken = $response['next_page_token'] ?? '';
        } while (!empty($nextPageToken));

        return $locations;
    }

    /**
     * Retrieves the details of a Zoom room location using location ID.
     *
     * @param string $locationId The ID of the Zoom room location.
     * @return array List of Zoom room locations
     */
    public function getZoomRoomLocationById($locationId)
    {
        if (empty($locationId)) {
            return null;
        }

        $accessToken = $this->credentialsService->getCurrentAccessToken();
        $response = $this->apiService->getApiResponse($accessToken, 'Zoom Locations', '/rooms/locations/' . $locationId);


        return $response ?? null;
    }

    /**
     * Retrieves the devices of a Zoom room using the Zoom API.
     *
     * @param int $roomId The ID of the Zoom room.
     * @return array An array of devices in the Zoom room.
     */
    public function getZoomRoomDevices($roomId)
    {
        if (empty($roomId)) {
            return [];
        }

        $accessToken = $this->credentialsService->getCurrentAccessToken();
        $response = $this->apiService->getApiResponse($accessToken, "Zoom Room Devices", "/rooms/$roomId/devices");


        if ($response && isset($response['devices'])) {
            return $response['devices'];
        }

        return [];
    }

    /**
     * Gets the room information from the given array.
     *
     * @param array $room The array containing API response.
     * @return array The parsed room data.
     */
    public function extractRoomDetails($room, $allZoomLocations)
    {
        $roomData = [
            'room_id' => $room['room_id'] ?? null,
            'zoom_location_id' => $room['location_id'] ?? null,
            'room_name' => $room['name'] ?? '',
            'room_status' => $room['status'] ?? null,
        ];

        $location = $this->zoomRepository->getTeqtivityLocationByRoomName($roomData['room_name']);
        $roomData['location_id'] = $location ? $location->id : null;

        // Get the site code and region from the Zoom location.
        $roomData = $this->getAdditionalDataFromZoomLocation($allZoomLocations, $roomData);

        // Auto create teqtivity location if it doesn't exist.
        if (!$location) {
            $this->createLocationHistory($roomData);
            $locationId = $this->createTeqtivityLocation($roomData);
            $roomData['location_id'] = $locationId;
        }

        return $roomData;
    }

    /**
     * Parses the Zoom API response and prepares the data for insertion into the DB.
     *
     * @param array $roomData The array containing room information.
     * @param array $deviceResponse The array containing device information.
     * @return array The parsed data.
     */
    public function parseZoomData($roomData, $deviceResponse)
    {
        $roomData['device_id']     = $deviceResponse['id'] ?? null;
        $roomData['serial_no']     = $deviceResponse['serial_number'] ?? null;
        $roomData['device_type']   = $deviceResponse['device_type'] ?? null;
        $roomData['device_system'] = $deviceResponse['device_system'] ?? null;
        $roomData['status']        = $deviceResponse['status'] ?? null;
        $roomData['hostname']      = $deviceResponse['device_hostname'] ?? null;
        $roomData['manufacturer']  = $deviceResponse['device_manufacturer'] ?? null;
        $roomData['model']         = $deviceResponse['device_model'] ?? null;
        $roomData['firmware']      = $deviceResponse['device_firmware'] ?? null;
        $roomData['app_version']   = $deviceResponse['app_version'] ?? null;
        $roomData['ip_address']    = $deviceResponse['ip_address'] ?? null;
        $roomData['mac_addresses'] = $this->getMacAddresses($deviceResponse);
        $roomData['asset_id']      = $this->zoomRepository->getAssetIdBySerialNo($roomData['serial_no']);

        return $roomData;
    }

    /**
     * Retrieves the MAC addresses from the device information.
     *
     * @param array $device The array containing device information.
     * @return string|null A comma-separated string of MAC addresses if available, null otherwise.
     */
    public function getMacAddresses($device)
    {
        if (empty($device)) {
            return null;
        }

        if (!isset($device['device_mac_addresses'])) {
            return null;
        }

        if (is_array($device['device_mac_addresses'])) {
            return implode(',', $device['device_mac_addresses']);
        }

        return $device['device_mac_addresses'];
    }

    /**
     * Retrieves additional data (country, site code, floor, building) from a Zoom location by traversing up the parent locations.
     *
     * @param array $allZoomLocations The array containing all Zoom locations.
     * @param array $zoomData The array containing room information.
     * @return array The parsed data with additional information.
     */
    public function getAdditionalDataFromZoomLocation($allZoomLocations, $zoomData)
    {
        $parentLocationId = $zoomData['zoom_location_id'] ?? null;

        if (!$parentLocationId) {
            return $zoomData;
        }

        while ($parentLocationId) {
            $parentLocation = $this->getZoomLocationById($parentLocationId, $allZoomLocations);

            if (!$parentLocation) {
                break;
            }

            $this->getAdditionalData($parentLocation, $zoomData);
            $parentLocationId = $parentLocation['parent_location_id'] ?? null;
        }

        return $zoomData;
    }

    /**
     * Given a location ID, returns the associated Zoom location from the given array of all Zoom locations.
     *
     * @param int $locationId The ID of the Zoom location.
     * @param array $allZoomLocations The array of all Zoom locations.
     * @return array|null The associated Zoom location or null if not found.
     */
    public function getZoomLocationById($locationId, $allZoomLocations)
    {
        if (empty($allZoomLocations)) {
            return null;
        }

        return collect($allZoomLocations)
            ->firstWhere('id', $locationId);
    }

    /**
     * Extracts additional data (country, site code, floor, building) from a Zoom location.
     *
     * @param array $parentLocation The Zoom location to extract data from.
     *
     * @return array The extracted data.
     */
    public function getAdditionalData($parentLocation, &$zoomData)
    {
        $type = $parentLocation['type'] ?? null;
        $name = $parentLocation['name'] ?? null;

        if ($name == "") {
            return;
        }

        if ($type === 'country') {
            $countryValSplitted = explode('/', $name); // country value come like NL/AMS/Tribes

            if ($countryValSplitted) {
                $zoomData['country'] = $countryValSplitted[0];
                $zoomData['site_code'] = $countryValSplitted[1];
            }
        }

        if ($type == 'floor') {
            $zoomData['floor'] = $name;
        }

        if ($type == 'building') {
            $zoomData['building'] = $name;
        }
    }

    /**
     * Creates a Teqtivity location record in the database.
     *
     * @param array $roomData The room data containing room_id, room_name, country,
     *                        site_code, floor, and building information.
     *
     * @return int The ID of the newly created location.
     */
    public function createTeqtivityLocation($roomData)
    {
        $locationTypeId = Cache::remember('location_type_install_id', 60 * 60, function () {
            return LocationType::where('location_type_name', 'Install')->first()->id;
        });

        $data = [
            'resource_id'      => $roomData['room_id'],
            'room_name'        => $roomData['room_name'],
            'country_id'       => Country::where('country_code', $roomData['country'] ?? '')->value('id'),
            'location_type_id' => $locationTypeId,
            'site_code_id'     => SiteCode::where('code', $roomData['site_code'] ?? '')->value('id'),
            'floor'            => $roomData['floor'] ?? '',
            'building'         => $roomData['building'] ?? '',
            'status'           => 1,
            'zoom'             => 'Zoom'
        ];

        return Location::create($data)->id;
    }

    /**
     * Create a history record when a location is created from a Zoom resource.
     *
     * @param array $roomData The Zoom resource item.
     *
     * @return boolean
     */
    private function createLocationHistory($roomData)
    {
        $description = __('history.LocationCreated', [
            'roomName' => $roomData['room_name']
        ]);

        $assetHistory = [
            'action' => 'location_added',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'created_by' => 'Zoom',
        ];
        event(new BulkUpdates($assetHistory));

        return true;
    }

    /**
     * Updates the asset ID associated with a Zoom device in the database.
     *
     * @param array $discoveryData The discovery data array containing the device ID.
     * @param int $assetId The asset ID to associate with the device.
     *
     * @return void
     */
    public function updateDiscoveryAssetId($discoveryData, $assetId)
    {
        ZoomDevice::where('device_id', $discoveryData['device_id'])->update(['asset_id' => $assetId]);
    }

    /**
     * Re-installs the location discrepancy assets from the Zoom repository.
     *
     * Queries the Zoom repository for all location discrepancy assets and calls
     * the reInstallDiscrepancyAssets method with the results.
     *
     * @return void
     */
    public function reInstallLocationDiscrepancyAssets()
    {
        $locationDiscrepancyAssets = $this->zoomRepository->getLocationDiscrepancy()->has('location')->get();
        $this->reInstallDiscrepancyAssets($locationDiscrepancyAssets, 'Zoom');
    }
}
