<?php

namespace App\Services\Zoom;

use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class ZoomHardwareDataDownloadService
{
    /**
     * Constructs a new instance of the ZoomDataDownloadService class.
     *
     * @param ZoomHardwareSyncService $syncService The ZoomHardwareSyncService instance used for API integration.
     */
    public function __construct(protected ZoomHardwareSyncService $syncService) {}

    /**
     * Get Zoom rooms API data and save response as JSON file.
     *
     * @return string The path to the saved JSON file.
     */
    public function getZoomRoomsDataAsJsonFile()
    {
        $data = [];

        try {
            $rooms = $this->syncService->syncZoomRooms(true);

            if (!$rooms) {
                return false;
            }

            foreach ($rooms as $key => $room) {
                $devices = $this->syncService->getZoomRoomDevices($room['room_id'] ?? null);
                $rooms[$key]['devices'] = $devices;

                $location = $this->syncService->getZoomRoomLocationById($room['location_id'] ?? null);
                $rooms[$key]['location'] = $location;

                sleep(5);
            }

            $data['rooms'] = $rooms;
            $fileName = 'zoom_rooms_' . now()->format('Y_m_d_H_i_s') . '.json';

            return $this->storeAsJsonFile($data, $fileName);
        } catch (Exception $e) {
            Log::channel('daily')->error('Error generating Zoom rooms JSON file: ' . $e->getMessage() . $e->getTraceAsString());

            return false;
        }
    }

    /**
     * Get Zoom rooms API data and save response as JSON file.
     *
     * @return string The path to the saved JSON file.
     */
    public function getZoomLocationsDataAsJsonFile()
    {
        $data = [];

        try {
            $locations = $this->syncService->getAllZoomRoomLocations();
            $data['locations'] = $locations;
            $fileName = 'zoom_locations_' . now()->format('Y_m_d_H_i_s') . '.json';

            return $this->storeAsJsonFile($data, $fileName);
        } catch (Exception $e) {
            Log::error('Error generating Zoom Locations JSON file: ' . $e->getMessage() . $e->getTraceAsString());

            return false;
        }
    }

    /**
     * Store the provided data as a JSON file.
     *
     * @param mixed $data The data to be stored as JSON.
     * @param string $fileName The name of the file to be saved.
     *
     * @return string The file name.
     */
    public function storeAsJsonFile($data, $fileName = 'zoom_data.json')
    {
        $filePath = 'export_reports/' . $fileName;

        Storage::disk('public')->put($filePath, json_encode($data));

        return $fileName;
    }
}
