<?php

namespace App\Services\Zoom;

use App\Repositories\ApiCredentialRepository;
use App\Services\Integrations\ZoomHardwareIntegration;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ZoomHardwareCredentialsService
{
    /**
     * Constructs a new instance of the ZoomService class.
     *
     * @param ApiCredentialRepository $apiCredentialRepo The ApiCredentialRepository instance.
     * @param ZoomHardwareIntegration $integrationService The ZoomHardwareIntegration instance.
     *
     * @return void
     */
    public function __construct(
        protected ApiCredentialRepository $apiCredentialRepo,
        protected ZoomHardwareIntegration $integrationService
    ) {}

    /**
     * Store API credentials in the database.
     *
     * @param array $data The request array containing the client ID and client secret.
     *
     * @return bool
     */
    public function storeApiCredentials($data)
    {
        try {
            $credentialsData = [
                'api_name'  => 'Zoom Hardware',
                'slug'      => 'zoom_hardware',
                'user_name' => $data['app_name'] ?? null,
                'key'       => $data['client_id'] ?? null,
                'password'  => $data['client_secret'],
            ];
            $this->apiCredentialRepo->create($credentialsData);

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->error('Zoom - credential create : ' . $e->getMessage());

            return false;
        }
    }

    /**
     * Updates the API credentials for the Zoom service.
     *
     * @param array $request The HTTP request object containing the updated client ID and client secret.
     * @param int $id The ID of the API credential to update.
     *
     * @return bool
     */
    public function updateApiCredentials($request, $id)
    {
        $credentials = $this->apiCredentialRepo->findById($id);

        if (!$credentials) {
            return false;
        }

        try {
            $updateData = [
                'user_name' => $request['app_name'] ?? $credentials->user_name,
                'key'       => $request['client_id'] ?? $credentials->key,
                'password'  => $request['client_secret'] ?? $credentials->password,
            ];
            $credentials->update($updateData);
            Cache::forget('zoom-hardware-credentials');

            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Clears the API token from the database.
     *
     * @param $credentials The API credential object.
     *
     * @return void
     */
    public function clearTokens($credentials)
    {
        $credentials->update(['data' => '']);
        Cache::forget('zoom-hardware-credentials');
    }

    /**
     * Checks the API connection.
     *
     * @return bool Returns true if the connection is successful, false otherwise.
     */
    public function checkApiConnection()
    {
        $accessToken = $this->getCurrentAccessToken();

        $response = $this->integrationService->getApiResponse($accessToken, 'Zoom - check connection', '/rooms');

        return  $response  ? true : false;
    }

    /**
     * Retrieves the current access token based on the provided credentials.
     * This method also checks if the access token is expired or not. If it is expired, it will refresh the access token.
     *
     * @return string|null The current access token, or null if not found.
     */
    public function getCurrentAccessToken()
    {
        $credentials = $this->getApiCredentials();

        if (!$credentials || !$credentials->data) {
            return null;
        }

        $tokenData = json_decode($credentials->data);

        // Check if token exist
        if (empty($tokenData->access_token) || empty($tokenData->expires_at)) {
            return null;
        }

        //Refresh access token if expired
        if ($this->isTokenExpired($tokenData->expires_at)) {
            $data = $this->integrationService->generateAccessToken($credentials);
            $this->storeAccessToken($data, $credentials);

            return $data['access_token'] ?? null;
        }

        return $tokenData->access_token;
    }

    /**
     * Retrieves the API credentials for the Zoom service.
     *
     * @return \App\Models\ApiCredential|null The API credentials for the Zoom service, or null if not found.
     */
    public function getApiCredentials()
    {
        return Cache::remember('zoom-hardware-credentials', now()->addMinutes(60), function () {
            return $this->apiCredentialRepo->getCredentials('zoom_hardware');
        });
    }

    /**
     * Check if the access token has expired.
     *
     * @param int $tokenExpiresAt The timestamp when the access token expires.
     * @return bool True if the access token has expired, false otherwise.
     */
    protected function isTokenExpired($tokenExpiresAt)
    {
        return now()->timestamp >= $tokenExpiresAt;
    }

    /**
     * Store the access token in the database.
     *
     * @param array $accessDetails The access details.
     * @param \App\Models\ApiCredential $zoomCredentials The API credential.
     * @return void
     */
    public function storeAccessToken($accessDetails, $zoomCredentials)
    {
        if (!$accessDetails) {
            return false;
        }

        $accessDetails['expires_at'] = $this->calculateExpirationTimestamp($accessDetails['expires_in']);
        $zoomCredentials->update(['data' => json_encode($accessDetails)]);

        Cache::forget('zoom-hardware-credentials');
    }

    /**
     * Calculate the expiration timestamp based on the provided expiration time.
     *
     * This method calculates the expiration timestamp by subtracting 15 minutes (900 seconds) from the expiration time to manage the API sync.
     *
     * @param mixed $expiresIn The expiration time in seconds.
     * @return int|null The expiration timestamp or null if the expiration time is invalid.
     */
    public function calculateExpirationTimestamp($expiresIn)
    {
        if (($expiresIn === null) || !is_numeric($expiresIn)) {
            return null;
        }

        $expiryTime = time() + $expiresIn;
        $expiryTime = $expiryTime - 900;

        return $expiryTime;
    }

    /**
     * Sets the connection status by generating an access token and updating the API credential data.
     *
     * @return bool Returns true if the connection is successful, false otherwise.
     */
    public function setConnection()
    {
        try {
            $credentialsData = $this->getApiCredentials();

            if (!$credentialsData) {
                return false;
            }

            $generatedAccessToken = $this->integrationService->generateAccessToken($credentialsData);

            if ($generatedAccessToken) {
                $this->storeAccessToken($generatedAccessToken, $credentialsData);
            }

            return $this->checkApiConnection();
        } catch (Exception $e) {
            Log::channel('daily')->error("Zoom API connection error-" . $e->getMessage());

            return false;
        }
    }
}
