<?php

namespace App\Services\Vendor;

use App\Events\BulkUpdates;
use App\Models\VendorCategory;
use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class VendorCategoryService
{

    public function __construct(protected CommonFilterRepository $commonFilterRepository, protected ReportOutputData $reportOutputData) {}

    /**
     * Get the vendor categories based on search
     * @return array of vendor categories and count
     */
    public function data()
    {
        $vendors = VendorCategory::whereNotNull('name');
        $inputData = $this->getInputData(request('form'));
        $vendors = $this->filter($vendors, $inputData);
        $count = $vendors->count();
        $vendors = $this->reportOutputData->getOutputData($vendors, ['name' => 'asc']);

        return compact('vendors', 'count');
    }

    /**
     * Returns the input form data
     * @param mixed $formData
     * 
     * @return [type]
     */
    private function getInputData($formData)
    {
        return [
            'category' => $formData['category'] ??  '',
        ];
    }

    /**
     * Apply filters to the vendors query based on input data.
     *
     * @param Builder $vendors The vendors query builder.
     * @param array   $inputData The input data for filtering.
     *
     * @return Builder The filtered vendors query builder.
     */
    private function filter($vendors, $inputData)
    {
        return $this->commonFilterRepository->filterWithDirectFields($vendors, 'id', $inputData['category']);
    }

    /**
     * Retrieves the vendor data, starting from a given index, and appending nested data for each vendor.
     *
     * @param array $vendors The list of vendors to process.
     * @param int   $start   The starting index.
     * @param array $data    The initial data array to which the nested data will be appended.
     *
     * @return array The updated data array with nested vendor data.
     */
    public function getData($vendors, $start, $data)
    {
        $parentIndex = $start;

        foreach ($vendors as $vendor) {
            $parentIndex++;
            $nestedData = $this->getNestedData($vendor, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }


    /**
     * Retrieves nested data for a given vendor and index.
     *
     * @param mixed $vendor The vendor object containing vendor details.
     * @param int $index The index of the vendor.
     *
     * @return array The nested data including vendor ID, name, and action links.
     */
    public function getNestedData($category, $index)
    {

        $nestedData['id'] = $index;
        $nestedData['name'] = $category->name;
        $nestedData['action'] = "<div class='btn-toolbar'><a href=''  data-toggle='modal' data-category-id='" . $category->id . "' data-target='#editCategoryModal' class='btn btn-link'><i class='icon icon-n-edit'></i></a>";
        if ($category->vendors->count() == 0) {
            $nestedData['action'] .= "<a href='javascript:void(0)' class='btn btn-link remove' data-category-id='" . $category->id . "' data-category-name='" . $category->name . "'><i class='icon icon-delete-forever'></i></a>";
        }
        $nestedData['action']   .= "</div>";

        return $nestedData;
    }

    /**
     * Creates an asset history with the given action and description.
     *
     * @param string $action The action to log in the asset history.
     * @param string $description The description to log in the asset history.
     *
     * @return void
     */
    public function createAssetHistory($action, $description)
    {
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => $action,
            'description' => $description,
            'created_at' => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at' => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
    }
}
