<?php

namespace App\Services\User;

use App\Http\Responses\ReportOutputData;
use App\Repositories\UserRepository;
use Illuminate\Support\Str;

class UserData
{

    /**
     * Constructor  OF User data class
     * @param  protected
     * @param  protected
     */
    public function __construct(protected UserRepository $userRepository, protected  ReportOutputData $reportOutputData)
    {
    }

    /**
     * Get a list of users after filtering.
     *
     * @return array
     */
    public function getData()
    {
        $users = $this->userRepository->getUsers();

        $inputData = $this->getInputData(request('form'));

        $users = $this->filter($users, $inputData);

        $count = $users->count('users.id');

        $users = $this->reportOutputData->getOutputData($users, ['first_name' => 'asc', 'last_name' => 'asc', 'email' => 'asc']);

        return compact('users', 'count');
    }

    /**
     * Get input data for filtering users.
     *
     * @param array $formData
     * @return array
     */
    public function getInputData($formData)
    {
        return [
            'searchText'                => $formData['searchText'] ?? '',
            'systemRole'                => $formData['system_role'] ?? '',
            'noOfAssets'                => $formData['no_of_assets'] ?? '',
            'dateFrom'                  => $formData['date_from'] ?? '',
            'dateTo'                    => $formData['date_to'] ?? '',
            'haveAssets'                => $formData['have_assets'] ?? '',
            'status'                    => $formData['status'] ?? '',
            'legalHold'                 => $formData['legalhold'] ?? '',
            'workLocation'              => $formData['work_location'] ?? '',
            'country'                   => $formData['country_id'] ?? '',
            'employeeType'              => $formData['employee_type'] ?? '',
            'employeeId'                => $formData['employee_id'] ?? '',
            'hasMobilePhoneAssigned'    => $formData['has_mobile_phone_assigned'] ?? '',
            'terminationDateFrom'       => $formData['term_date_from'] ?? '',
            'terminationDateTo'         => $formData['term_date_to'] ?? '',
        ];
    }

    /**
     * Filter users based on input data.
     *
     * @param $users
     * @param array $inputData
     * @return mixed
     */
    public function filter($users, $inputData)
    {
        $users = $this->userRepository->searchUserWithGeneralQueries($users, $inputData['searchText']);
        $users = $this->userRepository->searchUserWithSystemRole($users, $inputData['systemRole']);
        $users = $this->userRepository->searchUserWithNumberOfAssets($users, $inputData['noOfAssets']);
        $users = $this->userRepository->searchUserWithToDate($users, $inputData['dateTo']);
        $users = $this->userRepository->searchUserWithFromDate($users, $inputData['dateFrom']);
        $users = $this->userRepository->searchUserThatHaveAssets($users, $inputData['haveAssets']);
        $users = $this->userRepository->searchUserWithStatus($users, $inputData['status']);
        $users = $this->userRepository->searchUserWithLegalHold($users, $inputData['legalHold']);
        $users = $this->userRepository->searchUserWithWorkLocation($users, $inputData['workLocation']);
        $users = $this->userRepository->searchUserWithCountry($users, $inputData['country']);
        $users = $this->userRepository->searchUserWithEmployeeType($users, $inputData['employeeType']);
        $users = $this->userRepository->searchUserWithEmployeeID($users, $inputData['employeeId']);
        $users = $this->userRepository->searchUserWithMobilePhoneAssigned($users, $inputData['hasMobilePhoneAssigned']);
        $users = $this->userRepository->searchUserWithTerminationDate($users, $inputData['terminationDateFrom'], $inputData['terminationDateTo']);

        return $users;
    }

    /**
     * Get user data for export.
     *
     * @param $users
     * @param $start
     * @param $data
     * @return array
     */
    public function getUserData($users, $start, $data)
    {
        $parentIndex = $start;

        foreach ($users as $key => $user) {

            $parentIndex++;

            $nestedData = $this->getNestedData($user, $parentIndex, '', '');

            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Get nested data for export.
     *
     * @param $user
     * @param $index
     * @param $last
     * @param $parent
     * @return array
     */
    public function getNestedData($user, $index, $last, $parent)
    {
        $email = route('users.show', $user->id);

        $nestedData['dataLast'] = $last;
        $nestedData['dataId'] = $user->id;
        $nestedData['dataParent'] = $parent;
        $nestedData['id'] = $index;

        $commonData = $this->getExportNestedData($user);
        foreach ($commonData as $key => $value) {
            $nestedData[Str::slug($key, '_')] = $value;
        }
        $nestedData['user_email'] = "<a href=$email>$user->email</a>";
        $nestedData['action'] = "<a class='btn btn-link' href=users/" . $user->id . "/edit><i class='icon icon-n-edit'></i></a>";

        return $nestedData;
    }

    /**
     * Get users for export.
     *
     * @param array $requestData
     * @return mixed
     */
    public function getExportData($requestData = array())
    {
        if (!empty($requestData)) {
            request()->merge($requestData);
        }
        $users = $this->userRepository->getUsers();

        $inputData = $this->getInputData(request()->all());

        $users = $this->filter($users, $inputData);

        $users = $users->orderByRaw('first_name asc, last_name asc, email asc');

        return $users;
    }

    /**
     * Get user data for export.
     *
     * @param $users
     * @param $start
     * @param $data
     * @return array
     */
    public function getExportUserData($users, $start, $data)
    {
        $parentIndex = $start;

        foreach ($users as $key => $user) {
            $parentIndex++;
            $nestedData = $this->getExportNestedData($user);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Get nested data for export.
     *
     * @param $user
     * @return array
     */
    public function getExportNestedData($user)
    {
        if (!empty($user->hire_date && $user->hire_date != '0000-00-00' && $user->hire_date != '11/30/-0001')) {
            $hireDate = parse_date_from_db_date(date('Y-m-d', strtotime($user->hire_date)));
        } else {
            $hireDate = '';
        }
        if (!empty($user->terminated_date && $user->terminated_date != '0000-00-00' && $user->terminated_date != '11/30/-0001')) {
            $terminatedDate = parse_date_from_db_date(date('Y-m-d', strtotime($user->terminated_date)));
        } else {
            $terminatedDate = '';
        }
        $nestedData['First Name'] = $user->first_name;
        $nestedData['Last Name'] = $user->last_name;
        $nestedData['Email'] = $user->email;
        $nestedData['Title'] = $user->position ? $user->position->name : '';
        $nestedData['Employee Id'] = $user->employee_id;
        $nestedData['User Type'] = $user->employeeType ? $user->employeeType->name : '';
        $nestedData['User Role'] = optional($user->userType)->name;
        $nestedData['Hire Date'] = $hireDate;
        $nestedData['Department'] = $user->department ? $user->department->name : '';
        $nestedData['Work Location'] = $user->city;
        $nestedData['Country']   = $user->country ? $user->country->name : '';
        $nestedData['Status'] = $user->status == 1 ? 'Active' : 'Terminated';
        $nestedData['Termination Date'] = $terminatedDate;
        // $nestedData['Legal Hold'] = $user->legalhold == 1 ? 'Yes' : 'No';
        $nestedData['# of Assets'] = $user->assets_count;

        return $nestedData;
    }
}
