<?php

namespace App\Services\User;

use App\Rules\BulkUploadDateFormat;
use Facades\App\Repositories\BulkUpload;
use Illuminate\Support\Facades\Validator;

class UserBulkUpload
{
    public function importAssetData($path)
    {
        $count = 1;
        $items = $this->getItems($path);

        //if (empty($items)) return false;
        if (empty($items) || empty($items['csvData'])) {
            return ['error' => 'Please make sure the file contains data'];
        }

        $requiredHeaders = $this->getHeaderMap();
        $fileHeaderMismatch = $this->isFileHeadersMismatch($items, $requiredHeaders);
        if ($fileHeaderMismatch) {
            return ['error' => $fileHeaderMismatch];
        }

        $csvData = [];
        foreach ($items['csvData'] as $key => $item) {

            if (!count($item)) continue; //skip empty columns

            $csvData[] = $this->generateCSVData($item, $count++);
        }

        return $csvData;
    }

    public function getItems($path)
    {
        if (!file_exists($path)) return false;
        $items = csv_to_array($path);
        if (empty($items)) return false;
        return $items;
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.userData');
    }

    /**
     * Checks if the uploaded CSV file has headers that match the required headers.
     * @param array $data The data returned from the CSV reader.
     * @param array $requiredHeaders The required headers.
     * @return boolean True if the CSV file headers does not match the required headers, false otherwise.
     */
    public function isFileHeadersMismatch($data, $requiredHeaders)
    {
        $requiredFileheaders = array_values($requiredHeaders);
        $uploadedFileheaders = $data['headers'];
        $messages = $this->checkFileDifference($uploadedFileheaders, $requiredFileheaders);
        if ($messages != '') {
            return $messages;
        }
    }

    /**
     * Check csv file difference from the template file
     * @param  array $csvKeys
     * @param  array $headerKeys
     * @return string
     */
    public function checkFileDifference(array $csvKeys, array $headerKeys)
    {
        if (count($headerKeys) >= count($csvKeys)) {
            $result = array_diff($headerKeys, $csvKeys);
            $messageString = 'Please make sure the column names are spelled correctly. Columns missing - ';
        } else {
            $result = array_diff($csvKeys, $headerKeys);
            $messageString = 'Uploaded file is not match with template file. Extra columns - ';
        }
        $messages = '';

        if (!empty($result)) {
            foreach ($result as $value) {
                $messageString .= $value . ', ';
            }
            $messages = rtrim($messageString, ", ");
        }
        return $messages;
    }

    public function generateCSVData($item, $count)
    {

        $headerMap = Config('bulk-upload.userData');

        if (empty($item)) return false;

        $userData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $userData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;
        }

        $userData['count_add'] = $count++;

        return compact('userData');
    }

    public function getRelationalValues($item, $dbField, $fileHeader)
    {
        if (in_array($dbField, ["department", "employee_type"])) {
            return $item[$fileHeader];
        }

        if ($dbField == "location_id") {
            return BulkUpload::getLocationId($item, $dbField, $fileHeader);
        }

        // if ($dbField == "position_id") {
        //     return BulkUpload::getPositionId($item, $dbField, $fileHeader);
        // }

        // if ($dbField == "department_id") {
        //     return BulkUpload::getDepartmentId($item, $dbField, $fileHeader);
        // }

        // if ($dbField == "employee_type_id") {
        //     return BulkUpload::getEmployeeTypeId($item, $dbField, $fileHeader);
        // }

        if ($dbField == "user_type_id") {
            return BulkUpload::getUserTypeId($item, $dbField, $fileHeader);
        }

        if ($dbField == 'is_rehire') {
            return BulkUpload::getIsRehireValue($item, $dbField, $fileHeader);
        }

        return false;
    }

    public function validateFields($csvData)
    {
        $count = 2;
        foreach ($csvData as $data) {
            $errors[] = $this->csvValidator($data['userData'], $count++);
        }
        return $errors;
    }

    public function csvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                'first_name'    => 'required',
                'last_name'     => 'required',
                'email'         => 'required|unique:users,email',
                'employee_id'   => 'required',
                'hire_date'     => ['required', new BulkUploadDateFormat($data['hire_date'], 'm/d/Y', $count, 'Hire')],
                'terminated_date'     => new BulkUploadDateFormat($data['terminated_date'], 'm/d/Y', $count, 'Terminated'),
            ],
            $messages = [
                'first_name.required'         => 'Line no ' . $count . ' : First name is required.',
                'last_name.required'         => 'Line no ' . $count . ' : Last name is required.',
                'email.required'         => 'Line no ' . $count . ' : Email required.',
                'hire_date.required'         => 'Line no ' . $count . ' : Hire date is required.',
                'email.unique'         => 'Line no ' . $count . ' : The email is already taken.',
                'employee_id.required'   => 'Line no ' . $count . ' : Employee Id required.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }
}
