<?php

namespace App\Services\Terminations\TermsReport;

use App\Repositories\TermAssets;
use Carbon\Carbon;

class TermsReportExport
{
    protected $termAssetsRepo;

    public function __construct(TermAssets $termAssetsRepo)
    {
        $this->termAssetsRepo = $termAssetsRepo;
    }

    public function filterExportData()
    {
        $inputData = $this->getExportInputData();
        $users = $this->termAssetsRepo->getTermAssets();

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'id', $inputData['userId']);

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'employee_id', $inputData['employeeId']);

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'employee_type_id', $inputData['employeeType']);

        $users = $this->termAssetsRepo->searchUserWithNumberOfAssets($users, $inputData['noOfAssets']);

        $users = $this->termAssetsRepo->searchUserWithLegalHold($users, $inputData['legalHold']);

        $users = $this->termAssetsRepo->searchUserWithTicketNo($users, $inputData['ticketNo']);

        $users = $this->termAssetsRepo->searchUserWithHasTicket($users, $inputData['hasTicket']);

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'city', $inputData['workLocation']);

        $users = $this->termAssetsRepo->searchUserWithOutstandingDays($users, $inputData['outstandingDays']);

        $users = $this->termAssetsRepo->searchUserWithFromDate($users, $inputData['dateFrom']);

        $users = $this->termAssetsRepo->searchUserWithToDate($users, $inputData['dateTo']);

        return $users;
    }

    public function getExportInputData()
    {
        return [
            'userId' => (request('employee_name')) ? request('employee_name') : '',
            'dateFrom' => (request('date_from')) ? request('date_from') : '',
            'dateTo' => (request('date_to')) ? request('date_to') : '',
            'employeeId' => (request('employee_id')) ? request('employee_id') : '',
            'employeeType' => (request('employee_type')) ? request('employee_type') : '',
            'ticketNo' => (request('ticket_no')) ? request('ticket_no') : '',
            'legalHold' => (request('legal_hold')) ? request('legal_hold') : '',

            'hasTicket' => (request('has_ticket')) ? request('has_ticket') : '',
            'noOfAssets' => (request('no_of_assets')) ? request('no_of_assets') : '',
            'workLocation' => (request('work_location')) ? request('work_location') : '',
            'outstandingDays' => (request('outstanding_days')) ? request('outstanding_days') : '',
            'fte'            => (request('fte')) ? request('fte') : '',
        ];
    }

    public function getExportUserData($users, $start, $data)
    {
        $parentIndex = $start;
        $data = [];
        foreach ($users as $key => $user) {
            ++$parentIndex;
            $userAssetData = $this->getAssetsUncollectedData($user, $parentIndex);
            foreach ($userAssetData as $key => $userAsset) {
                $data[] = $userAsset;
            }
        }

        return $data;
    }

    public function getAssetsUncollectedData($user, $index)
    {
        $userAssetData = [];
        for ($i = 0; $i < $user->term_asset_count; ++$i) {
            $nestedData = $this->getExportNestedData($user, $i);
            $userAssetData[] = $nestedData;
        }

        return $userAssetData;
    }

    public function getExportNestedData($user, $i)
    {
        $nestedData['Employee ID'] = disableCSVInjection($user->employee_id);
        $nestedData['First Name'] = disableCSVInjection($user->first_name);
        $nestedData['Last Name'] = disableCSVInjection($user->last_name);
        $nestedData['Email'] = disableCSVInjection($user->email);
        $nestedData['Employee Type'] = disableCSVInjection($user->employeeType ? $user->employeeType->name : '');
        $nestedData['Department'] = disableCSVInjection($user->department ? $user->department->name : '');
        $nestedData['Work Location'] = disableCSVInjection($user->city);

        $nestedData['Legal Hold'] = disableCSVInjection(($user->legalhold == 0) ? 'No' : 'Yes');
        $nestedData['User Status'] = disableCSVInjection(($user->status == 0) ? 'Terminated' : 'Active');

        $terminated_date = $user->terminated_date;
        if ($terminated_date && $terminated_date !== '11/30/-0001') {
            $terminated = new Carbon($terminated_date);
            $now = Carbon::now();
            $nestedData['Terminated Date'] = disableCSVInjection($user->terminated_date);
            $nestedData['Outstanding days'] = disableCSVInjection($terminated->diff($now)->days);
        } else {
            $nestedData['Terminated Date'] = '';
            $nestedData['Outstanding days'] = '';
        }


        $termAsset = $user->termAsset[$i];
        $nestedData['Serial #'] = disableCSVInjection($termAsset->serial_no);
        $nestedData['Asset Tag #'] = disableCSVInjection($termAsset->asset_tag);
        $nestedData['Hardware Standard '] = disableCSVInjection($termAsset->makeAndModel ? $termAsset->makeAndModel->makeModelName : '');
        $nestedData['Tech Specs'] = disableCSVInjection($termAsset->technicalSpec ? $termAsset->technicalSpec->details : '');

        $nestedData['Assets Uncollected'] = disableCSVInjection($user->term_asset_count);
        $nestedData['Ticket Number'] = disableCSVInjection(optional($user->userJiraMappings)->ticket_no);
        $nestedData['Ticket Status'] = disableCSVInjection(optional($user->userJiraMappings)->ticket_status);

        $nestedData['Tracking Number']  = $this->trackingNumberButton($user);

        $nestedData['Shipment Status']  = $this->getShipmentStatus($user);

        $comments = array();
        if ($user->termAssetsComment) {
            foreach ($user->termAssetsComment as $key => $comment) {
                $comments[]     = $comment->comment . " By " . $comment->addedBy->email . " at " . $comment->created_at->diffForHumans();
            }
        }
        $nestedData['Comment'] = disableCSVInjection(implode(',', array_filter($comments)));

        return $nestedData;
    }

    public function trackingNumberButton($user)
    {
        $shipment = optional($user->assets)->first();
        return optional($shipment->assetTracking)->tracking_number ?? '';
    }

    public function getShipmentStatus($user)
    {
        $shipment = optional($user->assets)->first();
        return optional($shipment->assetTracking)->shipment_status;
    }
}
