<?php

namespace App\Services\Terminations\TermsReport;

use App\Repositories\TermAssets;
use App\Repositories\UserRepository;
use Carbon\Carbon;

class TermsReportData
{
    protected $termAssetsRepo;
    protected $userRepository;

    public function __construct(TermAssets $termAssetsRepo, UserRepository $userRepository)
    {
        $this->termAssetsRepo = $termAssetsRepo;
        $this->userRepository = $userRepository;
    }

    public function filter()
    {
        $inputData = $this->getInputData();
        $users = $this->termAssetsRepo->getTermAssets();

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'id', $inputData['userId']);

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'employee_id', $inputData['employeeId']);

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'employee_type_id', $inputData['employeeType']);

        $users = $this->termAssetsRepo->searchUserWithNumberOfAssets($users, $inputData['noOfAssets']);

        $users = $this->termAssetsRepo->searchUserWithLegalHold($users, $inputData['legalHold']);

        $users = $this->termAssetsRepo->searchUserWithTicketNo($users, $inputData['ticketNo']);

        $users = $this->termAssetsRepo->searchUserWithHasTicket($users, $inputData['hasTicket']);

        $users = $this->termAssetsRepo->searchUserWithUserFields($users, 'city', $inputData['workLocation']);

        $users = $this->termAssetsRepo->searchUserWithOutstandingDays($users, $inputData['outstandingDays']);

        $users = $this->termAssetsRepo->searchUserWithFromDate($users, $inputData['dateFrom']);

        $users = $this->termAssetsRepo->searchUserWithToDate($users, $inputData['dateTo']);

        if ($inputData['fte'] && $inputData['fte'] == 'full_time') {
            $users = $this->userRepository->getFteUsers($users, $inputData['fte']);
        }
        if ($inputData['fte'] && $inputData['fte'] == 'ext') {
            $users = $this->userRepository->getExtUsers($users, $inputData['fte']);
        }

        $count = $users->count('id');

        $users = $this->getOutputData($users);

        return compact('users', 'count');
    }

    public function getInputData()
    {
        return [
            'userId' => isset(request('form')['employee_name']) ? request('form')['employee_name'] : '',
            'dateFrom' => isset(request('form')['date_from']) ? request('form')['date_from'] : '',
            'dateTo' => isset(request('form')['date_to']) ? request('form')['date_to'] : '',
            'employeeId' => isset(request('form')['employee_id']) ? request('form')['employee_id'] : '',
            'employeeType' => isset(request('form')['employee_type']) ? request('form')['employee_type'] : '',
            'ticketNo' => isset(request('form')['ticket_no']) ? request('form')['ticket_no'] : '',
            'legalHold' => isset(request('form')['legal_hold']) ? request('form')['legal_hold'] : '',
            'hasTicket' => isset(request('form')['has_ticket']) ? request('form')['has_ticket'] : '',
            'noOfAssets' => isset(request('form')['no_of_assets']) ? request('form')['no_of_assets'] : '',
            'workLocation' => isset(request('form')['work_location']) ? request('form')['work_location'] : '',
            'outstandingDays' => isset(request('form')['outstanding_days']) ? request('form')['outstanding_days'] : '',
            'fte'            => isset(request('form')['fte']) ? request('form')['fte'] : '',
        ];
    }

    public function getOutputData($users)
    {
        $start = request('start');
        $limit = request('length');
        if ($limit != -1) {
            $users = $users->offset($start)
                ->limit($limit);
        }
        $users = $users->orderBy('terminated_date', 'DESC');

        return $users->get();
    }

    public function getUserData($users, $start, $data)
    {
        $parentIndex = $start;

        foreach ($users as $key => $user) {
            ++$parentIndex;
            $nestedData = $this->getNestedData($user, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    public function getNestedData($user, $index)
    {
        $userTerminatedDate = '';
        if (!empty($user->terminated_date && $user->terminated_date != '0000-00-00' && $user->terminated_date != '11/30/-0001')) {
            $userTerminatedDate = $user->terminated_date;
        }
        $nestedData['id'] = $index;
        $nestedData['name'] = $user->first_name . ' ' . $user->last_name;
        $nestedData['employee_id'] = $user->employee_id;
        $nestedData['email'] = $user->email;
        $nestedData['employee_type'] = $user->employeeType ? $user->employeeType->name : '';
        $nestedData['department'] = $user->department ? $user->department->name : '';
        $nestedData['work_location'] = $user->city;
        $nestedData['legal_hold'] = ($user->legalhold == 0) ? 'No' : 'Yes';
        $nestedData['status'] = ($user->status == 0) ? 'Terminated' : 'Active';
        $nestedData['terminated_date'] = $userTerminatedDate ? Carbon::parse($userTerminatedDate)->format('m/d/Y') : '';

        $terminated = $userTerminatedDate ? new Carbon($userTerminatedDate) : '';
        $now = Carbon::now();
        $nestedData['difference'] = $terminated ? ($terminated->diff($now)->days) : '';

        $nestedData['uncollected'] = $user->term_asset_count;
        if ($nestedData['uncollected'] > 0) {
            $nestedData['uncollected'] = "<a href='' data-action='" . url('terms-report/get-term-assets/' . $user->id) . "'  data-toggle='modal' data-term-asset-id='" . $user->id . "' data-target='#viewAssetModal'>" . $nestedData['uncollected'] . "</a>";
        }
        $nestedData['ticket_no'] = optional($user->userJiraMappings)->ticket_no;
        $nestedData['ticket_status'] = optional($user->userJiraMappings)->ticket_status;
        $commentString = $action = '';

        if (count($user->termAssetsComment) > 0) {
            $commentString = "<a href='' data-toggle='modal' data-term-asset-id='" . $user->id . "' data-target='#commentModel'  data-tooltip='tooltip' data-placement='left' title='View Comments' class='btn btn-link'><i class='icon icon-comment-add'></i></a>";
        }
        $updateUrl = route('termAssetUpdateComment', ['id' => $user->id]);
        $action = "<a href='' data-action='" . $updateUrl . "' data-toggle='modal' data-term-asset-id='" . $user->id . "' data-target='#addModal' title='Add Comment' class='btn btn-link' data-tooltip='tooltip' data-placement='left' title='Add Comment'><i class='icon icon-n-edit'></i></a>";

        $nestedData['comment'] = $action . '  ' . $commentString;

        $nestedData['tracking_number']  = $this->trackingNumberButton($user);

        $nestedData['shipment_status']  = $this->getShipmentStatus($user);

        return $nestedData;
    }

    public function trackingNumberButton($user)
    {
        $shipment = optional($user->assets)->first();
        $trackingNumber = optional($shipment->assetTracking)->tracking_number ?? '#';
        return "<a href='' class='btn asset-tracking-btn' data-toggle='modal' data-tracking-number='" . optional($shipment->assetTracking)->tracking_number . "' data-user='" . $user->id . "'> " . $trackingNumber . " </a>";
    }

    public function getShipmentStatus($user)
    {
        $shipment = optional($user->assets)->first();
        return optional($shipment->assetTracking)->shipment_status;
    }
    /**
     * Update asset button for assets in termination report
     *
     * @param $asset App\Models\Asset
     */
    public function getUpdateAssetButton($asset)
    {
        return '<a href="javascript:;" data-asset="' . $asset->id . '" data-asset-serial="' . $asset->serial_no . '" data-asset-current-status="' . optional($asset->AssetStatus)->slug . '" data-parent-asset="' . $asset->parent_asset_id . '" class="btn btn-primary update-terminated-user-asset">Update Status</a>';
    }

    /**
     * Get all uncollected assets of a user
     * @param $userId
     * @return $uncollectedAssetsArray
     */

    public function getAllUncollectedUserAssets($userId)
    {
        return  $this->termAssetsRepo->getAllUncollectedUserAssets($userId);
    }
    /**
     * Create uncollected assets table for view
     * @param $termAssets
     * @return $tableView
     */

    public function getUncollectedAssetViewData($termAssets)
    {
        $tableBody = '';
        foreach ($termAssets as $key => $termAsset) {
            $ticket_no =  $this->termAssetsRepo->getUserAssignmentTicket($termAsset->user->id, $termAsset->id);
            $tableBody .= '<tr>
                            <td>' . $termAsset->serial_no . '</td>
                            <td>' . $termAsset->asset_tag . '</td>
                            <td>' . ($termAsset->makeAndModel ? $termAsset->makeAndModel->name : '') . '</td>
                            <td>' . ($termAsset->assetStatus ? $termAsset->assetStatus->name : '') . '</td>
                            <td>' . $ticket_no . '</td>
                            <td>' . date('m/d/Y', ($termAsset->latestAssetHistory ? strtotime($termAsset->latestAssetHistory->updated_at) : strtotime($termAsset->updated_at->toDateString()))) . '</td>
                            <td>' . $this->getUpdateAssetButton($termAsset) . '</td>
                        </tr>';
        }

        return $tableBody;
    }
}
