<?php

namespace App\Services\Terminations\Retriever;

use App\Models\RetriverShipmentRequest;
use App\Services\Integrations\Retriever\RetrieverIntegrationService;
use Illuminate\Support\Facades\Auth;

class RetrieverShipmentService
{
    /**
     * Constructs a new instance of the RetrieverShipmentService class which handles the shipment request operations.
     *
     * @param RetrieverIntegrationService $retrieverService The retriever service used for integration.
     */
    public function __construct(protected RetrieverIntegrationService $retrieverService)
    {
    }

    /**
     * Checks the connection to the API by calling the `checkApiConnection` method of the `retrieverService` object.
     *
     * @return bool Returns `true` if the connection is successful, `false` otherwise.
     */
    public function checkConnection()
    {
        return $this->retrieverService->checkApiConnection();
    }

    /**
     * Creates a shipment request.
     *
     * @param array $shipmentData The data for the shipment request.
     * @throws Exception If there is an error creating the shipment request.
     * @return bool Returns true if the shipment request is created successfully, false otherwise.
     */
    public function createShipmentRequest($shipmentData)
    {
        $status = false;
        $shipmentData['requester_id'] = Auth::user()->id;
        $shipmentData['return_charger'] = (isset($shipmentData['return_charger']) === TRUE) ? 1 : 0;
        $shipmentData['return_phone'] =  (isset($shipmentData['return_phone']) === TRUE) ? 1 : 0;

        $shipmentId = RetriverShipmentRequest::create($shipmentData)->id;

        $shipmentRequestData = $this->getShipmentRequestData($shipmentData);

        $response = $this->retrieverService->createShipmentRequest($shipmentRequestData);

        if ($response->status() === 201) {
            $status = true;
        }

        RetriverShipmentRequest::where('id', $shipmentId)->update(['request_status' => $status]);

        return ['status' => $status, 'errors' => $response->object()];
    }

    /**
     * Retrieves the shipment request data from the given request data.
     *
     * @param array $requestData The request data.
     *
     * @return array The shipment request data.
     */
    public function getShipmentRequestData($requestData)
    {
        return [
            'request_charger' => $requestData['return_charger'],
            'request_cell_phone' => $requestData['return_phone'],
            'device' => $requestData['device'],
            'request_disposal' => $requestData['request_disposal'],
            'employee_info'       => [
                'email'           => $requestData['email'],
                'name'            => $requestData['name'],
                'address_line_1'  => $requestData['address_line_1'],
                'address_line_2'  => $requestData['address_line_2'],
                'address_city'    => $requestData['address_city'],
                'address_state'   => $requestData['address_state'],
                'address_country' => $requestData['address_country'],
                'address_zip'     => $requestData['address_zip']
            ],
            'company_info' => [
                'return_recipient_name'  => $requestData['return_recipient_name'],
                'return_address_company' => $requestData['return_address_company'],
                'return_address_line_1'  => $requestData['return_address_line_1'],
                'return_address_line_2'  => $requestData['return_address_line_2'],
                'return_address_city'    => $requestData['return_address_city'],
                'return_address_state'   => $requestData['return_address_state'],
                'return_address_country' => $requestData['return_address_country'],
                'return_address_zip'     => $requestData['return_address_zip'],
                'display_name'           => $requestData['display_name'],
                'notification_email'     => $requestData['notification_email']
            ]
        ];
    }
}
