<?php

namespace App\Services\Terminations\Retriever;

use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\Location;
use App\Models\RetrieverDevice;
use App\Services\Integrations\Retriever\RetrieverIntegrationService;
use App\User;
use Carbon\Carbon;
use Facades\App\Services\AssetHistory as AssetHistoryService;

class RetrieverDataSyncService
{

    /**
     * Constructor for the RetrieverDataSyncService class accepting the RetrieverIntegrationService object.
     *
     * @param RetrieverIntegrationService $integrationService The integration service for Retriever.
     */
    public function __construct(protected RetrieverIntegrationService $integrationService)
    {
    }

    /**
     * Checks the connection to the Retriever API by calling the checkApiConnection method of the integration service.
     *
     * @return bool Returns true if the API connection is successful, false otherwise.
     */
    public function checkConnection()
    {
        return $this->integrationService->checkApiConnection();
    }

    /**
     * Synchronizes the retriever devices by retrieving them from the API and saving them to the database.
     *
     * @param array $devices The array of devices retrieved from the API.
     * @return int The total number of synced devices.
     */
    public function syncRetrieverDevices()
    {
        $nextPageLink = '';
        $totalSyncedCount = 0;

        // RetrieverDevice::truncate();

        do {
            $response = $this->integrationService->getRetrieverDevices($nextPageLink);

            if ($response && isset($response->results)) {
                $totalSyncedCount += $this->saveRetrieverDevices($response->results);
            }

            $nextPageLink =  $response ? $response->next : null;
        } while ($nextPageLink != null);

        return $totalSyncedCount;
    }

    /**
     * Save the Return Devices details to DB
     * 
     * @param Array $devices    API response devices details
     */
    public function saveRetrieverDevices($devices)
    {
        $counter = 0;
        foreach ($devices as $device) {
            $data = $this->getDataToSave($device);
            RetrieverDevice::create($data);
            // if ($data['shipment_status'] == "return_complete") { 
            //     $returnLocation = optional($device->company_info)->return_address_state;
            //     $result = $this->updateAssetStatus($data, $returnLocation);
            // }
            $counter++;
        }

        return $counter;
    }

    /**
     * Retrieves the data to save from the given device.
     *
     * @param mixed $device The device object.
     * @return array The merged data containing basic information, shipment information, and timestamps.
     */
    public function getDataToSave($device)
    {
        $userId = $this->getUserId($device->employee_info->email);
        $assetId = 0; // Assuming assetId is set to 0 for now

        return array_merge(
            $this->getBasicInfo($device, $userId, $assetId),
            $this->getShipmentInfo($device),
            $this->getTimestamps($device)
        );
    }

    /**
     * Retrieves the user ID associated with the given email.
     *
     * @param string $email The email address to search for.
     * @return int The user ID if found, 0 otherwise.
     */
    private function getUserId($email)
    {
        if ($email) {
            $user = User::select('id')->where('personal_email', $email)->first();
            if (!$user) {
                $user = User::select('id')->where('email', $email)->first();
            }
            return $user ? $user->id : 0;
        }
        return 0;
    }

    /**
     * Retrieves basic information about a device.
     *
     * @param mixed $device The device object.
     * @param int $userId The user ID.
     * @param int $assetId The asset ID.
     * @return array The basic information of the device.
     */
    private function getBasicInfo($device, $userId, $assetId)
    {
        return [
            'user_id' => $userId,
            'asset_id' => $assetId,
            'serial_no' => $device->id, // Actually it's not serial no. it's a unique ID of the return order
            'user_email' => $device->employee_info->email ?? null,
            'user_name' => $device->employee_info->name ?? null,
            'city' => $device->employee_info->address_city ?? null,
        ];
    }

    /**
     * Retrieves the shipment information from the given device.
     *
     * @param object $device The device object containing the shipments.
     * @return array The shipment information including shipment status, device type,
     *               request charger, outbound tracking, outbound carrier, return tracking,
     *               return carrier, and deactivated reason.
     */
    private function getShipmentInfo($device)
    {
        if (isset($device->shipments[0])) {
            $shipment = $device->shipments[0];
            return [
                'shipment_status' => $shipment->status,
                'device_type' => $shipment->device_type,
                'request_charger' => $shipment->request_charger,
                'outbound_tracking' => $shipment->outbound_tracking,
                'outbound_carrier' => $shipment->outbound_carrier,
                'return_tracking' => $shipment->return_tracking,
                'return_carrier' => $shipment->return_carrier,
                'deactivated_reason' => $shipment->deactivated_reason,
            ];
        }
        return [
            'shipment_status' => null,
            'device_type' => null,
            'request_charger' => null,
            'outbound_tracking' => null,
            'outbound_carrier' => null,
            'return_tracking' => null,
            'return_carrier' => null,
            'deactivated_reason' => null,
        ];
    }

    /**
     * Retrieves the timestamps for a given device.
     *
     * @param object $device The device object.
     * @return array The timestamps for the device, including 'box_shipped_at', 'box_delivered_at',
     *               'device_shipped_at', 'device_delivered_at', and 'created_at'. If the device has no
     *               shipments, all timestamps will be set to null except for 'created_at', which will be set
     *               to null.
     */
    private function getTimestamps($device)
    {
        if (isset($device->shipments[0])) {
            $shipment = $device->shipments[0];
            return [
                'box_shipped_at' => format_date_iso_datetime_to_db_datetime_v2($shipment->box_shipped_at),
                'box_delivered_at' => format_date_iso_datetime_to_db_datetime($shipment->box_delivered_at),
                'device_shipped_at' => format_date_iso_datetime_to_db_datetime($shipment->device_shipped_at),
                'device_delivered_at' => format_date_iso_datetime_to_db_datetime($shipment->device_delivered_at),
                'created_at' => format_date_iso_datetime_to_db_datetime_v2($device->created_at),
            ];
        }
        return [
            'box_shipped_at' => null,
            'box_delivered_at' => null,
            'device_shipped_at' => null,
            'device_delivered_at' => null,
            'created_at' => null,
        ];
    }

    /**
     * Updates status of assets based on Retriver shipment status and creates history for it.
     * 
     * @param array $data response data
     * @param string $returnLocation Retriver shipment return location
     * 
     * @return void
     */
    public function updateAssetStatus($data, $returnLocation)
    {
        $retainHoldStatus = AssetStatus::whereSlug('retain_hold')->first();
        $user = User::select('id')->with('assets.assetStatus')->find($data['user_id']);
        if ($user) {
            foreach ($user->assets as $asset) {
                if (!$this->canUpdateAssetStatus($asset, $retainHoldStatus->id)) {
                    continue;
                }

                $dataToBeUpdated = array();
                $oldStatus = $asset->assetStatus;
                $newLocationId = $this->getAssetLocation($returnLocation);
                $loanerRetentionDate = Carbon::now()->addDays(config('date.period.retain_hold'))->format('m/d/Y');

                $dataToBeUpdated = [
                    'asset_status_id' => $retainHoldStatus->id,
                    'location_id' => $newLocationId,
                    'loaner_retention_date' => $loanerRetentionDate,
                ];
                $assetHistory = $this->createAssetHistory($asset, $dataToBeUpdated, $retainHoldStatus, $oldStatus);

                $asset->update($dataToBeUpdated);
                AssetHistory::create($assetHistory);
            }
        }
    }

    /**
     * Retrives location id based on return location field in API response
     * 
     * @param string $returnLocation
     * 
     */
    public function getAssetLocation($returnLocation)
    {
        if ($returnLocation == "NY") {
            return optional(Location::whereRoomName('NY Storage Room')->first())->id;
        }

        if ($returnLocation == "CA") {
            return optional(Location::whereRoomName('SF IT Storage')->first())->id;
        }

        if ($returnLocation == "CO") {
            return optional(Location::whereRoomName('Denver IT Storage')->first())->id;
        }

        return null;
    }

    /**
     * Creates asset history data for given asset
     * 
     * @param Asset $asset
     * @param array $data new values for asset
     * @param AssetStatus $newStatus
     * @param AssetStatus $oldStatus
     * 
     * @return array
     */
    public function createAssetHistory($asset, $data, $newStatus, $oldStatus)
    {
        $description = AssetHistoryService::getStatusDescription($newStatus->name, $data, $asset);
        $assetHistory = [
            'asset_id' => $asset->id,
            'action' => 'status_updated',
            'old_value' => $oldStatus->name,
            'new_value' => $newStatus->name,
            'new_asset_status_id' => $newStatus->id,
            'old_asset_status_id' => $oldStatus->id,
            'new_location_id' => $data['location_id'] ?? '',
            'old_location_id' => $asset->location_id,
            'new_user_id' => $asset->user_id,
            'old_user_id' => $asset->user_id,
            'description' => $description,
            'created_by'   => config('retriever.display_name'),
        ];

        return $assetHistory;
    }

    /**
     * Checks if asset status can be updated by Retriever sync. 
     * This method returns true only if asset is not yet updated by Retriever API and not in retain hold status.
     * 
     * @param array $data Retriever response data
     * 
     * @return boolean
     */
    public function canUpdateAssetStatus($asset, $retainHoldStatusId)
    {
        if ($asset->asset_status_id === $retainHoldStatusId) {
            return false;
        }

        if ($asset->retrieverUpdateHistory) {
            return false;
        }

        return true;
    }
}
