<?php

namespace App\Services\Terminations\HrCollection;

use App\Repositories\TermAssets;
use App\User;
use Carbon\Carbon;

class UserStatisticsGraphService
{
    protected $repository;

    /**
     * Constructor for the class.
     *
     * @param TermAssets $repository The repository for term assets.
     */
    public function __construct(TermAssets $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Generates data array for user graph report.
     * Loops through each month and retrives no of terminated users & no of users returned assets for both current year and last year
     *
     * @return array
     */
    public function generateGraphData()
    {
        $monthNames = [];
        $currentYearData = [];
        $lastYearData = [];
        $currentYear = Carbon::now()->year;
        $lastYear = $currentYear - 1;

        for ($month = 1; $month < 13; $month++) {
            $currentYearDate = Carbon::createFromDate($currentYear, $month, 1);
            $currentYearFirstDay = $currentYearDate->format('m/d/Y');
            $currentYearLastDay = $currentYearDate->endOfMonth()->format('m/d/Y');
            $currentYearCount = $this->getCount($currentYearFirstDay, $currentYearLastDay);

            $lastYearDate = Carbon::createFromDate($lastYear, $month, 1);
            $lastYearFirstDay = $lastYearDate->format('m/d/Y');
            $lastYearLastDay = $lastYearDate->endOfMonth()->format('m/d/Y');
            $lastYearCount = $this->getCount($lastYearFirstDay, $lastYearLastDay);

            $currentYearData['terminatedUsers'][] = $currentYearCount['terminatedUsers'];
            $currentYearData['receivedAssets'][] = $currentYearCount['receivedAssets'];
            $lastYearData['terminatedUsers'][] = $lastYearCount['terminatedUsers'];
            $lastYearData['receivedAssets'][] = $lastYearCount['receivedAssets'];
            $monthNames[] = $currentYearDate->format('F');
        }

        return compact('monthNames', 'lastYearData', 'currentYearData');
    }

    /**
     * Returns no of terminated users & no of users returned assets for given date range & user type.
     * @param date $dateFrom
     * @param date $dateTo
     * @param string $type
     * @return array
     */
    public function getCount($dateFrom, $dateTo)
    {
        $users = User::inActive();
        $users = $this->repository->searchUserWithFromDate($users, $dateFrom);
        $users = $this->repository->searchUserWithToDate($users, $dateTo);
        $terminatedUsersCount = $users->count();
        $receivedAssetsCount = $users->withCount('receivedAsset')->having('received_asset_count', '>', 0)->count();

        $result['terminatedUsers'] = $terminatedUsersCount;
        $result['receivedAssets'] = $receivedAssetsCount;

        return $result;
    }
}
