<?php

namespace App\Services\Terminations\HrCollection;

use App\Repositories\Reports\RetrievalStatisticsRepository;
use Carbon\Carbon;

class RetrievalStatisticsReportGraphService
{
    protected $repository;

    /**
     * Constructor for the class.
     *
     * @param RetrievalStatisticsRepository $repository The repository for retrieval statistics.
     */
    public function __construct(RetrievalStatisticsRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Retrieves the graph data for the current and previous year.
     *
     * @return array Returns an array containing the months, current year data, and last year data.
     */
    public function getGraphData()
    {
        $year = Carbon::now()->year;
        $lastYear = Carbon::now()->year - 1;
        $currentYearData = [];
        $lastYearData = [];
        $months = [];

        for ($month = 1; $month < 13; $month++) {
            // current year data
            $date = Carbon::createFromDate($year, $month, 1);
            $firstDay = $date->format('Y-m-d');
            $lastDay = $date->endOfMonth()->format('Y-m-d');
            $months[] = $date->format('F');
            $assets = $this->getCount($firstDay, $lastDay);
            $currentYearData['assetsComingBack'][] = $assets['assetsComingBackCount'];
            $currentYearData['assetsLostStolen'][] = $assets['assetsLostStolenCount'];

            // last year data
            $lastYearDate = Carbon::createFromDate($lastYear, $month, 1);
            $lastYearFirstDay = $lastYearDate->format('m/d/Y');
            $lastYearLastDay = $lastYearDate->endOfMonth()->format('m/d/Y');
            $lastYearCount = $this->getCount($lastYearFirstDay, $lastYearLastDay);
            $lastYearData['assetsComingBack'][] = $lastYearCount['assetsComingBackCount'];
            $lastYearData['assetsLostStolen'][] = $lastYearCount['assetsLostStolenCount'];
        }

        return compact('months', 'year', 'currentYearData', 'lastYearData');
    }

    /**
     * Retrieves the count of lost/stolen assets and coming back assets
     * within a specified date range.
     *
     * @param datatype $fromDate The start date of the range.
     * @param datatype $toDate   The end date of the range.
     * @return array Returns an array with the count of assets coming back
     *               and the count of lost/stolen assets.
     */
    public function getCount($fromDate, $toDate)
    {
        // lost / stolen assets
        $assetsLostStolenQuery = $this->repository->getData();
        $assetsLostStolenCount = $this->repository->assetsLostStolen($assetsLostStolenQuery, $fromDate, $toDate)->count();
        // coming  back assets
        $assetsComingBackQuery = $this->repository->getData();
        $assetsComingBackCount = $this->repository->assetsComingBack($assetsComingBackQuery, $fromDate, $toDate)->count();

        return compact('assetsComingBackCount', 'assetsLostStolenCount');
    }
}
