<?php

namespace App\Services;

use App\Models\Location;
use App\Models\Carrier;
use App\Models\UserType;
use App\Models\AssetStatus;
use App\User;
use Cache;
use Illuminate\Support\Facades\Auth;

class StaticDataService
{
    /**
     * Get the static data needed in the search page
     * @param type $type 
     * @return type
     */
    public function getStaticSearchAssetData($type)
    {
        $data = [];

        $data['installLocations'] = $this->getCachedInstallLocations();
        $data['warehouseLocations'] = $this->getCachedStorageLocations();
        $data['installAndBuildingLocations'] = $this->getInstallBuildingLocations();
        $data['breadcrumbTitle'] = $this->getBreadCrumbTitle($type);

        $data['user'] = '';
        $data['location'] = '';
        $data['asset'] = '';

        $data['type'] = request('general_type');

        $data['searchedLocationAssetTagCount'] = 0;
        $data['searchedUserAssetTagCount'] = 0;

        if ($type == 'mobile_assets') {
            $data['carriers'] = Carrier::get();
        }

        $data['statuses'] = $this->getAssetStatus($type);

        $data['adminStatus'] = $this->getAdminStatus();

        return $data;
    }

    /**
     * Get the Cached Qube Locations
     * @return collection
     */
    public function getCachedInstallLocations()
    {
        return Cache::remember('install-locations', 43200, function () {
            return Location::withoutwarehouse()->active()->orderBy('room_name', 'asc')->get();
        });
    }

    /**
     * Get the Cached Storage Locations
     * @return collection
     */
    public function getCachedStorageLocations()
    {
        return Cache::remember('warehouse-locations', 43200, function () {
            return Location::warehouses()->active()->valid()->unfreezed()->orderBy('room_name', 'asc')->get();
        });
    }

    public function getInstallBuildingLocations()
    {
        return Cache::remember('install-building-locations', 43200, function () {
            return Location::installBuildings()->active()->orderBy('room_name', 'asc')->get();
        });
    }

    public function getBreadCrumbTitle(string $type)
    {
        if ($type == "it_assets") {
            return 'Manage IT Assets';
        }

        if ($type == "mobile_assets") {
            return 'Manage Mobile Assets';
        }

        if ($type == "network_assets") {
            return 'Manage Network Assets';
        }

        if ($type == "av_assets") {
            return 'Manage AV Assets';
        }

        return 'Manage Assets';
    }

    public function getAdminStatus()
    {
        $userTypes = ['Super User', 'Super Admin', 'Administrator'];
        $userTypeName = UserType::find(User::find(Auth::id())->user_type_id)->name;
        $adminStatus = 0;

        if (in_array($userTypeName, $userTypes)) {
            $adminStatus = 1;
        }

        return $adminStatus;
    }

    public function getUser($userId)
    {
        return Cache::remember('user-' . $userId, 43200, function () use ($userId) {
            return User::select('id', 'first_name', 'last_name', 'email', 'user_type_id')->find($userId);
        });
    }

    public function getAssetStatus($type)
    {
        if ($type == "av_assets" || $type == "network_assets" || $type == "research_assets") {
            return AssetStatus::where('slug', '!=', 'assigned')->orderBy('slug')->get();
        }

        return AssetStatus::orderBy('slug')->get();
    }
}
