<?php

namespace App\Services\SoftwareLicense;

use App\Models\SoftwareLicense\SoftwareLicenseSubscription;
use Illuminate\Support\Facades\Validator;
use App\Repositories\SoftwareLicense\LicenseRepository;
use App\Rules\SoftwareLicense\LicenseKeyFormat;
use App\Rules\SoftwareLicense\LicenseKeysCount;
use App\Services\SoftwareLicense\LicenseService;
use Illuminate\Validation\Rule;

class SoftwareAssetsBulkUpload
{

    protected $softwareLicenseRepo;
    protected $licenseService;

    /**
     * constructor for Software Assets Bulk Upload
     */
    public function __construct(LicenseRepository $softwareLicenseRepo, LicenseService $licenseService)
    {
        $this->softwareLicenseRepo = $softwareLicenseRepo;
        $this->licenseService = $licenseService;
    }


    /**
     * Import asset Data from the uploded csv
     *
     * @param string $path
     * @param int $count
     */
    public function importUploadData($path, $count)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];

        $csvColumnErrors = $this->validateCsvColumns($path, $items);

        if ($csvColumnErrors) {
            $csvData['error'] = $csvColumnErrors;

            return $csvData;
        }

        if (!empty($items)) {

            foreach ($items as $key => $item) {

                if (!count($item)) continue; //skip empty columns

                $count++;
                $singleData = $this->generateUploadSoftwareAssetData($item, $count);
                $csvData['csvData'][] = $singleData;
            }
        }
        return $csvData;
    }

    /**
     * Make csv records to array
     *
     * @param string $path
     */
    public function getItems(string $path)
    {
        if (!file_exists($path)) return false;
        $items = csv_to_array($path);

        if (empty($items)) return false;

        return $items;
    }

    /**
     * generate data from csv records to upload
     */
    public function generateUploadSoftwareAssetData($item, $count)
    {

        $headerMap = Config('software-license.softwareAssetData');

        if (empty($item)) return false;

        $assetData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {

            $relationalColumnValue = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($relationalColumnValue !== false) {
                $item[$fileHeader] = $relationalColumnValue;
            }


            $formattedColumnValue = $this->getFormattedValues($item, $dbField, $fileHeader);
            $item[$fileHeader] = $formattedColumnValue;


            $assetData[$dbField] = (!empty($item[$fileHeader]) || $item[$fileHeader] == 0) ? $item[$fileHeader] : null;
        }
        // dd($assetData);
        $assetData['count_add'] = $count++;
        session(['count_add' => $assetData['count_add']]);
        // dd($assetData);
        return compact('assetData');
    }

    /**
     * get relational values for the foreign key csv columns
     *
     * @param array $item
     * @param string $dbField
     * @param string $fileHeader
     *
     * @return mixed
     */
    public function getRelationalValues(array $item, string $dbField, string $fileHeader)
    {
        if ($dbField == "software_license_category_id") {
            return $this->softwareLicenseRepo->getSoftwareLicenseCategoryId($item, $dbField, $fileHeader);
        }

        if ($dbField == "vendor_id") {
            $vendorId = $this->softwareLicenseRepo->getVendorId($item, $dbField, $fileHeader);
            if ($item[$fileHeader] && !$vendorId) {
                $vendorId = 'invalidnotexist';
            }
            return $vendorId;
        }

        if ($dbField == "software_license_manufacturer_id") {
            return $this->softwareLicenseRepo->getSoftwareLicenseManufacturerId($item, $dbField, $fileHeader);
        }

        if ($dbField == "payment_method_id") {
            return $this->softwareLicenseRepo->getSoftwareLicensePaymentMethodId($item, $dbField, $fileHeader);
        }

        if ($dbField == "software_license_subscription_id") {
            return $this->softwareLicenseRepo->getSoftwareLicenseSubscriptionId($item, $dbField, $fileHeader);
        }

        return false;
    }

    /**
     * get formatted values for the software license options columns
     *
     * @param array $item
     * @param string $dbField
     * @param string $fileHeader
     *
     * @return mixed
     */
    public function getFormattedValues(array $item, string $dbField, string $fileHeader)
    {
        if ($dbField == "auto_renewal") {
            // dd($item[$fileHeader]);
            // dd(array_search($item[$fileHeader], config('software-license.auto_renewal')));
            return array_search($item[$fileHeader], config('software-license.auto_renewal')) ? array_search($item[$fileHeader], config('software-license.auto_renewal')) : 'n/a';
        }

        if ($dbField == "type") {
            if ($item[$fileHeader] == '') {
                return '';
            }

            $searchedType = strtolower(str_replace(" ", "", $item[$fileHeader]));
            $allTypes = config('software-license.types');
            array_walk($allTypes, function (&$value, $key) {
                $value = strtolower(str_replace(" ", "", $value));
            });
            return array_search($searchedType, $allTypes) ? array_search($searchedType, $allTypes) : 'na';
            // return array_search($item[$fileHeader], config('software-license.types')) ? array_search($item[$fileHeader], config('software-license.types')) : 'na';
        }

        if ($dbField == "license_key_type") {
            if ($item[$fileHeader] == '') {
                return '';
            }
            return array_search($item[$fileHeader], config('software-license.license_key_types')) ? array_search($item[$fileHeader], config('software-license.license_key_types')) : 'na';
        }

        if ($dbField == "key_usage_type") {
            if ($item[$fileHeader] == '') {
                return '';
            }
            return array_search($item[$fileHeader], config('software-license.key_usage_types')) ? array_search($item[$fileHeader], config('software-license.key_usage_types')) : 'na';
        }

        if ($dbField == "licenses_purchased") {
            if ($item['License Key (type)'] == 'user_license') {
                return $item[$fileHeader];
            }
            return '';
        }

        return $item[$fileHeader];
    }

    /**
     * Validate columns of csv
     *
     * @param string $path
     * @param array $csvData
     *
     * @return string
     */
    public function validateCsvColumns($path, $csvData)
    {
        $headerMap = Config('software-license.softwareAssetData');
        $headerKeys = array_values($headerMap);
        $csv = array_map("str_getcsv", file($path, FILE_SKIP_EMPTY_LINES));
        $csvKeys = array_shift($csv);

        if (empty($csvData)) {
            return 'Please make sure the file contains data';
        }
        $messages = $this->checkFileDifference($csvKeys, $headerKeys);

        return $messages;
    }

    /**
     * check CSV File columns and mapping array difference
     *
     * @param array $csvKeys
     * @param array $headerKeys
     *
     * @return string
     */
    public function checkFileDifference($csvKeys, $headerKeys)
    {
        if (count($headerKeys) >= count($csvKeys)) {
            $result = array_diff($headerKeys, $csvKeys);
            $messageString = 'Please make sure the column names are spelled correctly. Columns missing - ';
        } else {
            $result = array_diff($csvKeys, $headerKeys);
            $messageString = 'Uploaded file is not match with template file. Extra columns - ';
        }
        $messages = '';

        if (!empty($result)) {
            foreach ($result as $key => $value) {
                $messageString .= $value . ', ';
            }
            $messages = rtrim($messageString, ", ");
        }

        return $messages;
    }

    /**
     * validate fields on each record
     *
     * @param array $csvData
     *
     * @return array
     */
    public function validateFields($csvData)
    {
        $count = 2;

        foreach ($csvData as $data) {
            $errors[] = $this->csvValidator($data['assetData'], $count++);
        }

        return $errors;
    }

    /**
     * sub function for validate fields on each record
     *
     * @param array $data
     * @param int $count
     *
     * @return mixed
     */
    public function csvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                'software_license_category_id' => 'required',
                'software_license_manufacturer_id' => 'required',
                'vendor_id' => ['nullable', Rule::notIn(['invalidnotexist'])],
                'name' => 'required',
                'software_license_subscription_id' => [Rule::requiredIf(function () use ($data) {
                    return in_array($data['license_type'], ['Subscription']);
                })],
                'type' => ['required', Rule::in(array_keys(config('software-license.types')))],
                'license_type' => 'required|in:' . implode(',', config('software-license.license_types')),
                'billing_cycle' => 'required|in:' . implode(',', config('software-license.billing_cycles')),
                'license_key_type' => ['required', Rule::in(array_keys(config('software-license.license_key_types')))],
                'key_usage_type' => [Rule::requiredIf(function () use ($data) {
                    return in_array($data['license_key_type'], ['single_license', 'user_license']);
                }), in_array($data['license_key_type'], ['single_license', 'user_license']) ? Rule::in(array_keys(config('software-license.key_usage_types'))) : ''],
                'start_date' => ['required', 'date_format:' . config('date.formats.read_date_format')],
                'subscription_expiration' => $data['license_type'] == 'Subscription' ? 'nullable|after_or_equal:start_date|date_format:' . config('date.formats.read_date_format') : 'nullable',
                'cost' => 'nullable|numeric|min:0',
                'payment_method_id' => 'required',
                'status' => 'required|in:' . implode(',', config('software-license.licence_status')),
                'license_key' => [new LicenseKeyFormat('/^[a-zA-Z0-9,_\r\n -]+$/i'), ($data['license_key_type'] == 'user_license' && ($data['licenses_purchased'] || $data['licenses_purchased'] == 0)) ? new LicenseKeysCount($data['licenses_purchased'], $count) : ''],
                'licenses_purchased'  => ['nullable', 'numeric', 'min:0', Rule::requiredIf(function ()  use ($data) {
                    return in_array($data['license_key_type'], ['user_license']);
                })],
            ],
            [
                'software_license_category_id.required' => 'Line no ' . $count . ' : The license category is required',
                'software_license_manufacturer_id.required' => 'Line no ' . $count . ' : The license manufacturer  is required',
                'name.required' => 'Line no ' . $count . ' : The name is required',
                'software_license_subscription_id.required' => 'Line no ' . $count . ' : The Subscription length is required when license type is Subscription and should be one of ' . implode(',', SoftwareLicenseSubscription::limit(10)->pluck('name')->toArray()) . ' etc.',
                'subscription_length.in' => 'Line no ' . $count . ' : The subscription length should be empty or use values "1 Month", "3 Months", "6 Months" or "1 Year"',
                'license_type.required' => 'Line no ' . $count . ' : The license type is required',
                'license_type.in' => 'Line no ' . $count . ' : The license type should be "Subscription" or "Perpetual"',
                'billing_cycle.required' => 'Line no ' . $count . ' : The billing cycle is required',
                'billing_cycle.in' => 'Line no ' . $count . ' : The billing cycle should be "Monthly" or "Annual"',
                'start_date.required' => 'Line no ' . $count . ' : The contract start date is required',
                'start_date.date_format'    => 'Line no ' . $count . ' : The contract start date format is incorrect. Use ' . config('date.formats.read_date_format') . ' format',
                'cost.numeric' => 'Line no ' . $count . ' : The cost should be a number',
                'cost.min' => 'Line no ' . $count . ' : The cost should not be less than zero',
                'payment_method_id.required' => 'Line no ' . $count . ' : The payment method is required',
                'licenses_purchased.numeric' => 'Line no ' . $count . ' : The license purchased should be a number',
                'licenses_purchased.min' => 'Line no ' . $count . ' : The license purchased should not be less than zero',
                'status.required' => 'Line no ' . $count . ' : The status is required',
                'status.in' => 'Line no ' . $count . ' : The status should be "Active" or "Inactive"',
                'license_key_type.required' => 'Line no ' . $count . ' : The license key type is required',
                'license_key_type.in' => 'Line no ' . $count . ' : The license key type should be one of ' . implode(',', array_values(config('software-license.license_key_types'))),
                'key_usage_type.required' => 'Line no ' . $count . ' : The key usage type is required',
                'key_usage_type.in' => 'Line no ' . $count . ' : The key usage type should be one of ' . implode(',', array_values(config('software-license.key_usage_types'))),
                'type.required' => 'Line no ' . $count . ' : The type is required',
                'type.in' => 'Line no ' . $count . ' : The type should be one of ' . implode(',', array_values(config('software-license.types'))),
                'vendor_id.not_in' => 'Line no ' . $count . ' : The vendor is invalid',
                'licenses_purchased.required' => 'Line no ' . $count . ' : The # of licenses purchased is required',
                'subscription_expiration.after_or_equal' => 'Line no ' . $count . ' : Subscription expiration date must be a date after or equal to start date',
                'subscription_expiration.date_format'    => 'Line no ' . $count . ' : The contract subscription expiration date format is incorrect. Use ' . config('date.formats.read_date_format') . ' format',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }
}
