<?php

namespace App\Services\SoftwareLicense;

use App\Models\SoftwareLicense\SoftwareLicense;
use App\Repositories\SoftwareLicense\LicenseUserRepository;

/**
 * Software License non teqtivity User Service
 */
class NonTeqUserService
{
    protected $licenseUserRepo;
    /**
     * Constructor for Software License non teqtivity User Service
     *
     * @param object $licenseRepo
     */
    public function __construct(LicenseUserRepository $licenseUserRepo)
    {
        $this->licenseUserRepo = $licenseUserRepo;
    }


    /**
     * This function retrieves a class of integrated users based on a software asset ID(ex: dropbox_members, zoom_members).
     * @param softwareAssetId 
     * @return an instance of the non-Teqtivity user class
     */
    public function getIntegratedUsersClass($softwareAssetId)
    {
        $softwareAsset  = SoftwareLicense::find($softwareAssetId);
        $namespace      = "\App\Models\\";
        $nonTeqtivityUserClassName = $namespace . ($softwareAsset->name ? $softwareAsset->name . 'Member' : '');

        if (!class_exists($nonTeqtivityUserClassName)) {

            return false;
        }

        $nonTeqUser = new $nonTeqtivityUserClassName();

        return  $nonTeqUser->whereNull('user_id');
    }

    /**
     * This function retrieves non-TEQ users for a given software asset
     * 
     * @param softwareAssetId
     */
    public function getNonTeqtivityUsersCount($softwareAssetId)
    {
        $nonTeqUsers = $this->getIntegratedUsersClass($softwareAssetId);
        if (!$nonTeqUsers) {

            return 0;
        }

        return $nonTeqUsers->count();
    }

    /**
     * This function retrieves non-Teqtivity users associated with a given software license ID.
     * 
     * @param softwareAssetId The ID of a software license for which we want to retrieve non-Teqtivity users.
     * 
     * @return a collection 
     */
    public function getNonTeqtivityUsers()
    {
        $softwareAssetId = request('form')['id'] ?? request('id');
        $nonTeqUser = $this->getIntegratedUsersClass($softwareAssetId);
        if (!$nonTeqUser) {

            return false;
        }

        $nonTeqUser = $this->filter($nonTeqUser);
        $nonTeqUser = $nonTeqUser->orderBy('created_at', 'desc');

        return $nonTeqUser;
    }

    /**
     * This function filters non-TEQ users based on various input data options.
     * 
     * @param nonTeqUsers 
     */
    public function filter($nonTeqUsers)
    {
        $inputData = request('form') ??  request()->all();
        $columnsForDateRangeMatch = ['joined_on_from', 'joined_on_to'];
        $columnsForExactMatch = ['role', 'status'];

        foreach ($inputData as  $filterKey => $filterValue) {
            if ($filterValue != '') {
                if (in_array($filterKey, $columnsForExactMatch)) {
                    $nonTeqUsers = $this->licenseUserRepo->filterByExactMatch($nonTeqUsers, $filterKey, $filterValue);
                    continue;
                }
                if ($filterKey == 'user_status') {
                    $nonTeqUsers = $this->licenseUserRepo->filterByUserStatus($nonTeqUsers, $filterKey, $filterValue);
                    continue;
                }
                if (in_array($filterKey, $columnsForDateRangeMatch)) {
                    $nonTeqUsers = $this->licenseUserRepo->filterByDateRangeMatch($nonTeqUsers, $filterKey, $filterValue);
                    continue;
                }
                if ($filterKey == 'searchText') {
                    $nonTeqUsers = $this->licenseUserRepo->filterByUserTextMatch($nonTeqUsers, $filterValue);
                    continue;
                }
            }
        }

        return $nonTeqUsers;
    }

    /**
     * This function retrieves a limited number of non teq users based on the start and length parameters.
     * 
     * @param licenseUsers 
     * @return a collection of license non teq users based on the specified limit and start parameters.
     */

    public function getLicenseUsersByLimit($licenseNonTeqUsers)
    {
        $start = request('start');
        $limit = request('length');

        if ($limit != -1) {
            $licenseNonTeqUsers = $licenseNonTeqUsers->offset($start)->limit($limit);
        }
        $licenseNonTeqUsers = $licenseNonTeqUsers->get();

        return $licenseNonTeqUsers;
    }
    /**
     * Set up Software License non teqtivity users for listing page
     *
     * @param object $licenseNonUsers
     * @param mixed $start
     * @param array $data
     *
     * @return array
     */
    public function getNonTeqtivityUsersData(object $licenseNonUsers, $start, array $data)
    {
        $parentIndex = $start;

        foreach ($licenseNonUsers as $key => $licenseNonUser) {

            $parentIndex++;
            $nestedData = $this->getNonTeqtivityUsersNestedData($licenseNonUser, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Formatting Software License non teqtivity users  for listing page
     *
     * @param object $licenseNonUser
     * @param int $index
     *
     * @return array
     */
    public function getNonTeqtivityUsersNestedData(object $licenseNonUser, int $index)
    {
        $nestedData['id']           = $index;
        $nestedData['user']         = optional($licenseNonUser)->name;
        $nestedData['email']        = optional($licenseNonUser)->email;
        $nestedData['license_key']  = '';
        $nestedData['date_added']   = $licenseNonUser->joined_on ? parse_date_from_db_datetime($licenseNonUser->joined_on) : '';
        $nestedData['user_status']  = ucfirst(optional($licenseNonUser)->status);
        $nestedData['role']         = ucfirst(str_replace('_', ' ', optional($licenseNonUser)->role));
        $nestedData['avg_meeting_length']   = $this->getMeetingsDetailsButton($licenseNonUser);

        return $nestedData;
    }

    /**
     * Generates a button to view meeting details for a given license non-user.
     *
     * @param LicenseNonUser $licenseNonUser The license non-user object.
     * @return string The HTML code for the button.
     */
    private function getMeetingsDetailsButton($licenseNonUser)
    {
        return '<a href="' . route('member-hosted-meetings', ['userId' => $licenseNonUser->zoom_user_id]) . '" class="btn btn-sm btn-primary" title="View monthwise details" style="width: 100px;">' . optional($licenseNonUser)->averageMeetingDuration() . '</a>';
    }

    /**
     * This function returns the count of non-Teq users who joined within a specified date range fora given software asset.
     * 
     * @param softwareAssetId
     * @param endDate
     * @return int
     */
    public function getActiveNonTeqUsersCount($softwareAssetId, $endDate = null)
    {
        $result = $this->getIntegratedUsersClass($softwareAssetId);

        if (!$result) {
            return 0;
        }

        $result =  $result->where('status', 'active');

        if ($endDate != null) {
            $result = $result->whereDate('joined_on', '<=', $endDate);
        }

        return $result->count();
    }



    /**
     * Set up Software License non teqtivity users for export
     *
     * @param object $licenseNonUsers
     * @param mixed $start
     * @param array $data
     *
     * @return array
     */
    public function getExportNonTeqUserData(object $licenseNonTeqUsers,  array $data)
    {
        foreach ($licenseNonTeqUsers as $licenseNonTeqUser) {
            $nestedData = $this->getExportNonTeqUsersNestedData($licenseNonTeqUser);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Formatting Software License non teqtivity users  for export
     *
     * @param object $licenseNonUser
     * @param int $index
     *
     * @return array
     */
    public function getExportNonTeqUsersNestedData(object $licenseNonUser)
    {
        $softwareAsset      = SoftwareLicense::find(request('id') ?? null);
        $softwareAssetName  = optional($softwareAsset)->name ?? '';

        $nestedData['User']                         = disableCSVInjection(optional($licenseNonUser)->name);
        $nestedData['Email Address']                = disableCSVInjection(optional($licenseNonUser)->email);
        $nestedData['Date Added']                   = disableCSVInjection($licenseNonUser->joined_on ? parse_date_from_db_datetime($licenseNonUser->joined_on) : '');
        $nestedData[$softwareAssetName . ' Status']   = disableCSVInjection(ucfirst(optional($licenseNonUser)->status));
        $nestedData[$softwareAssetName . ' Role']     = disableCSVInjection(ucfirst(str_replace('_', ' ', optional($licenseNonUser)->role)));
        if ($softwareAssetName == 'Zoom') {
            $nestedData['Average Meeting Length']     = disableCSVInjection(optional($licenseNonUser)->averageMeetingDuration());
        }

        return $nestedData;
    }
}
