<?php

namespace App\Services\SoftwareLicense;

use App\Models\SoftwareLicense\SoftwareLicenseUser;
use App\Repositories\SoftwareLicense\LicenseRepository;
use App\Repositories\SoftwareLicense\LicenseUserRepository;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Services\SoftwareLicense\LicenseHistoryService;
use Illuminate\Support\Facades\DB;

/**
 * Software License User Service
 */
class LicenseUserService
{
    protected $licenseRepo;
    protected $licenseUserRepo;
    protected $licenseHistoryService;

    /**
     * Constructor for Software License user Service
     *
     * @param object $licenseRepo
     */

    public function __construct(LicenseRepository $licenseRepo, LicenseHistoryService $licenseHistoryService, LicenseUserRepository $licenseUserRepo)
    {
        $this->licenseRepo = $licenseRepo;
        $this->licenseHistoryService = $licenseHistoryService;
        $this->licenseUserRepo = $licenseUserRepo;
    }

    /**
     * Get Software License users data
     * @return QueryResult
     */

    public function getLicenseUsers()
    {
        $licenseId = request('form')['id'] ?? request('id');
        $licenseUsers = $this->licenseUserRepo->getLicenseUsers($licenseId);
        $licenseUsers = $this->filter($licenseUsers);
        $licenseUsers = $licenseUsers->orderBy('created_at', 'desc');

        return $licenseUsers;
    }

    /**
     * This function retrieves a limited number of license users based on the start and length parameters.
     * 
     * @param licenseUsers 
     * @return a collection of license users based on the specified limit and start parameters.
     */

    public function getLicenseUsersByLimit($licenseUsers)
    {
        $start = request('start');
        $limit = request('length');

        if ($limit != -1) {
            $licenseUsers = $licenseUsers->offset($start)->limit($limit);
        }
        $licenseUsers = $licenseUsers->get();

        return $licenseUsers;
    }

    /**
     * This function filters a list of license users based on various input data parameters.
     * 
     * @param licenseUsers 
     * @return QueryResult
     */

    public function filter($licenseUsers)
    {
        $inputData = request('form') ??  request()->all();
        $columnsForDateRangeMatch = ['joined_on_from', 'joined_on_to'];
        $columnsForExactMatch = ['user_id', 'status'];
        foreach ($inputData as  $filterKey => $filterValue) {
            if ($filterValue != '') {
                if (in_array($filterKey, $columnsForExactMatch)) {
                    $licenseUsers = $this->licenseUserRepo->filterByExactMatch($licenseUsers, $filterKey, $filterValue);
                    continue;
                }
                if ($filterKey == 'user_status') {
                    $licenseUsers = $this->licenseUserRepo->filterByUserStatus($licenseUsers, $filterKey, $filterValue);
                    continue;
                }
                if (in_array($filterKey, $columnsForDateRangeMatch)) {
                    $licenseUsers = $this->licenseUserRepo->filterByDateRangeMatch($licenseUsers, $filterKey, $filterValue);
                    continue;
                }
            }
        }

        return $licenseUsers;
    }

    /**
     * Set up Software License users for listing page
     *
     * @param object $licenseUsers
     * @param mixed $start
     * @param array $data
     *
     * @return array
     */

    public function getLicensesUserData(object $licenseUsers, $start, array $data)
    {
        $parentIndex = $start;

        foreach ($licenseUsers as $key => $licenseUser) {
            $parentIndex++;
            $nestedData = $this->getLicensesUserNestedData($licenseUser, $parentIndex);

            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Formatting Software License users for listing page
     *
     * @param object $licenseUser
     * @param int $index
     * 
     * @return array
     */

    public function getLicensesUserNestedData(object $licenseUser, int $index)
    {
        $nestedData['id']         = $index;
        $nestedData['user']   = optional($licenseUser->user)->first_name . " " . optional($licenseUser->user)->last_name;
        $nestedData['email']   = "<a href='/users/" . optional($licenseUser->user)->id . "' target='__blank'>" . optional($licenseUser->user)->email  . "</a>";
        $nestedData['license_key']   = optional($licenseUser->licenseKey)->license_key;
        $nestedData['date_added']    =  $licenseUser->joined_on ? parse_date_from_db_datetime($licenseUser->joined_on) : '';
        $nestedData['user_status'] = $licenseUser->user->status == 1 ? 'Active' : 'Terminated';
        $nestedData['status']      = ucfirst($licenseUser->status);
        $nestedData['avg_meeting_length']   = $this->getMeetingsDetailsButton($licenseUser);
        $nestedData['action']      = "<a class='delete-location' data-id='" . $licenseUser->id . "' data-toggle='modal' data-target='#deleteLicenseUserModal'><i class='icon icon-delete-forever'></i></a>";

        return $nestedData;
    }

    /**
     * Generates the HTML for the "Meetings Details" button.
     *
     * @param datatype $licenseUser The license user object.
     * @return string The HTML for the "Meetings Details" button.
     */
    private function getMeetingsDetailsButton($licenseUser)
    {
        if (optional($licenseUser->zoomMember)->zoom_user_id) {
            return '<a href="' . route('member-hosted-meetings', ['userId' => optional($licenseUser->zoomMember)->zoom_user_id]) . '" class="btn btn-sm btn-primary" title="View monthwise details" style="width: 100px;">' . optional($licenseUser->zoomMember)->averageMeetingDuration() . '</a>';
        }
        return '';
    }

    /**
     * Set up Software License users for export
     *
     * @param object $licenseUsers
     * @param mixed $start
     * @param array $data
     *
     * @return array
     */

    public function getExportUserData(object $licenseUsers, array $data)
    {
        foreach ($licenseUsers as $licenseUser) {
            $nestedData = $this->getExportUserNestedData($licenseUser);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Formatting Software License users for export
     *
     * @param object $licenseUser
     * @param int $index
     * 
     * @return array
     */

    public function getExportUserNestedData(object $licenseUser)
    {
        $licenceName = optional($licenseUser->license)->name;
        $nestedData['User']             = disableCSVInjection(optional($licenseUser->user)->userName);
        $nestedData['Email Address']            = disableCSVInjection(optional($licenseUser->user)->email);

        if (!in_array($licenceName, config('software-license.imported_softwares'))) {
            $nestedData['License Key']  = disableCSVInjection(optional($licenseUser->licenseKey)->license_key);
        }

        $nestedData['Date Added']       =  disableCSVInjection($licenseUser->joined_on ? parse_date_from_db_datetime($licenseUser->joined_on) : '');
        $nestedData['Teqtivity Status'] = disableCSVInjection($licenseUser->user->status == 1 ? 'Active' : 'Terminated');

        if (in_array($licenceName, config('software-license.imported_softwares'))) {

            $nestedData[$licenceName . ' Status']       = disableCSVInjection(ucfirst($licenseUser->status));
            if ($licenceName == 'Zoom') {
                $nestedData['Average Meeting Length']   = disableCSVInjection(optional($licenseUser->zoomMember)->averageMeetingDuration());
            }
        }


        return $nestedData;
    }

    /**
     * This function checks if a license user exists for a given user ID and license ID.
     * 
     * @param userId
     * @param licenseId
     * 
     * @return the count
     */
    public function checkLicenseUserExists($userId, $licenseId)
    {
        return $this->licenseRepo->getLicenseUser($userId, $licenseId)->count();
    }

    /**
     * Add User to Software License
     *
     * @param array $data
     */

    public function createLicenseUser($data)
    {
        DB::beginTransaction();
        try {
            $licenseData = $this->getLicense($data['software_license_id']);
            $this->licenseRepo->createLicenseUser($data);

            $data['software_license_key_id'] = isset($data['software_license_key_id']) ? $data['software_license_key_id'] : null;

            if ($data['software_license_key_id']) {
                $licenseKeyData = $this->getLicenseKey($data['software_license_key_id']);
                ($licenseKeyData->users_allowed_for_key != 'U') ? $licenseKeyData->increment('no_of_users') : '';
            }

            //increment licenses_used count
            $licenseData->increment('licenses_used');

            $description = $this->licenseHistoryService->getStatusDescriptionForUserActions('user_added', $data['user_id'], $licenseData, $data['software_license_key_id'] ? $licenseKeyData->license_key : '');
            $this->licenseRepo->addLicenseHistory([
                'software_license_id' => $data['software_license_id'],
                'action' => 'user_added',
                'description' => $description,
                'license_user_id' => $data['user_id'],
                'software_license_key_id' => $data['software_license_key_id'],
                'user_id' => Auth::id()
            ]);
            DB::commit();
            return true;
        } catch (Exception $e) {
            Log::error($e->getMessage());
            // dd($e->getMessage());
            DB::rollback();
            return false;
        }
    }

    /**
     * Remove User from Software License
     *
     * @param int $id
     * @param int $licenseId
     *
     * @return bool
     */
    public function removeUserFromLicense(int $id, int $licenseId)
    {
        try {
            $licenseData = $this->getLicense($licenseId);
            $licenseUserData = $this->licenseRepo->getUserFromLicenseData($id);
            $licenseKeyData = collect([]);
            if ($licenseUserData->software_license_key_id) {
                $licenseKeyData = $this->getLicenseKey($licenseUserData->software_license_key_id);
                ($licenseKeyData->users_allowed_for_key != 'U') ? $licenseKeyData->decrement('no_of_users') : '';
            }
            $this->licenseRepo->removeUserFromLicense($id);
            //decrement licenses_used count
            $licenseData->decrement('licenses_used');

            $description = $this->licenseHistoryService->getStatusDescriptionForUserActions('user_removed', $licenseUserData->user_id, $licenseData, $licenseUserData->software_license_key_id ? optional($licenseKeyData)->license_key : '');
            $this->licenseRepo->addLicenseHistory([
                'software_license_id' => $licenseId,
                'action' => 'user_removed',
                'description' => $description,
                'license_user_id' => $licenseUserData->user_id,
                'software_license_key_id' => $licenseUserData->software_license_id,
                'user_id' => Auth::id()
            ]);

            return true;
        } catch (Exception $e) {
            // dd($e->getMessage(),$e->getLine(),$e->getFile());
            Log::error($e->getMessage());

            return false;
        }
    }

    /**
     * get Software License data
     *
     * @param int $id
     *
     * @return object
     */
    public function getLicense(int $id)
    {
        return $this->licenseRepo->getLicense($id);
    }


    /**
     * get Software License key data
     *
     * @param int $id
     *
     * @return object
     */
    public function getLicenseKey(int $id)
    {
        return $this->licenseRepo->getLicenseKey($id);
    }

    /**
     * get count of licenced active users in
     * softwat asset with in a period
     */
    public function getActiveTeqUsersCount($softwareAssetId, $endDate = null)
    {
        $teqUsers = SoftwareLicenseUser::where(['software_license_id' => $softwareAssetId, 'status' => 'active']);

        if ($endDate != null) {
            $teqUsers = $teqUsers->whereDate('joined_on', '<=', $endDate);
        }

        return $teqUsers->count();
    }
}
