<?php

namespace App\Services\SoftwareLicense;

use App\Models\SoftwareLicense\SoftwareLicenseSubscription;
use App\Repositories\SoftwareLicense\LicenseRepository;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Services\SoftwareLicense\LicenseHistoryService;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Support\Facades\DB;

/**
 * Software License Service
 */
class LicenseService
{

    protected $licenseRepo;
    protected $licenseHistoryService;
    protected $nonTeqUserService;
    protected $teqUserService;
    /**
     * Constructor for Software License Service
     * 
     * @param object $licenseRepo
     */
    public function __construct(LicenseRepository $licenseRepo, LicenseHistoryService $licenseHistoryService,  NonTeqUserService $nonTeqUserService, LicenseUserService  $teqUserService)
    {
        $this->licenseRepo = $licenseRepo;
        $this->licenseHistoryService = $licenseHistoryService;
        $this->nonTeqUserService = $nonTeqUserService;
        $this->teqUserService = $teqUserService;
    }

    /**
     * List Software License Paymnet Methods
     * 
     * @return array
     */
    public function getAllPaymentMethods()
    {
        return $this->licenseRepo->getAllPaymentMethods()->get(['id', 'name']);
    }

    /**
     * Create Software License
     * 
     * @param array $data
     * 
     * @return mixed
     */
    public function createLicense($data)
    {

        try {
            $data = $this->formatLicenseData($data);

            $license = $this->licenseRepo->createLicense($data);
            $description = $this->licenseHistoryService->getStatusDescriptionForCreated($license);

            $this->licenseRepo->addLicenseHistory([
                'software_license_id' => $license->id,
                'action' => 'created',
                'description' => $description,
                'user_id' => Auth::id()
            ]);
            return $license;
        } catch (Exception $e) {
            // dd($e->getMessage());
            Log::error($e->getMessage());

            return false;
        }
    }

    /**
     * get renewal data dynamically with start date and subscription length
     * 
     * @param string $startDate
     * @param string $subscriptionLength
     * @param string $licenseType
     * 
     * @return mixed
     */
    public function getRenewalDate($startDate, $subscriptionLength, $licenseType)
    {
        $noRenwalDateSubcriptions = ['perpetual'];

        if (!$startDate || !$subscriptionLength || in_array($licenseType, $noRenwalDateSubcriptions)) {
            return null;
        }

        $subscriptionLengthData = SoftwareLicenseSubscription::find($subscriptionLength);

        $renewalDate = ($subscriptionLengthData->duration_type == 'Years' || $subscriptionLengthData->duration_type == 'Year') ? addYears($startDate, $subscriptionLengthData->duration) : addMonths($startDate, $subscriptionLengthData->duration);
        // dd($subscriptionLengthData,$subscriptionLength,$renewalDate,$startDate);
        return convert_to_db_date($renewalDate);
    }

    /**
     * Fetch Software License Categories with filter
     * 
     * @return array
     */
    public function filter()
    {
        $query = $this->licenseRepo->getAllLicenses();
        $query = $this->filterWithOptions($query);
        $start = request('start');
        $limit = request('length');
        $count = $query->count();

        if ($limit != -1) {
            $query = $query->offset($start)->limit($limit);
        }

        $query = $query->orderBy('created_at', 'desc');
        $licenses = $query->get();

        return compact('licenses', 'count');
    }

    /**
     * filter the list with the search options
     * 
     * @param object $query
     * 
     * @return object
     */
    public function filterWithOptions(object $query)
    {
        $searchValues = request('form');
        $columnsForExactMatch = [
            'software_license_category_id',
            'software_license_manufacturer_id',
            'licenses_purchased',
            'type',
            'license_key_type',
            'key_usage_type',
            'vendor_id',
            'license_type',
            'status',
            'licenses_used'
        ];
        $columnsForTextMatch = ['order_number', 'name'];
        $columnsForDateRangeMatch = ['renewal_date_from', 'renewal_date_to'];
        $columsForCountMatch = [];

        foreach ($searchValues as $filterKey => $filterValue) {

            if ($filterValue != '') {

                if (in_array($filterKey, $columnsForExactMatch)) {
                    $query = $this->licenseRepo->filterByExactMatch($query, $filterKey, $filterValue);
                }

                if (in_array($filterKey, $columnsForTextMatch)) {
                    $query = $this->licenseRepo->filterByTextMatch($query, $filterKey, $filterValue);
                }

                if (in_array($filterKey, $columnsForDateRangeMatch)) {
                    $query = $this->licenseRepo->filterByDateRangeMatch($query, $filterKey, $filterValue);
                }
                if (in_array($filterKey, $columsForCountMatch)) {
                    $query = $this->licenseRepo->filterByCountMatch($query, $filterKey, $filterValue);
                }
            }
        }

        return $query;
    }

    /**
     * Set up Software Licenses for listing page
     * 
     * @param object $licenses
     * @param mixed $start
     * @param array $data
     * 
     * @return array  
     */
    public function getLicensesData(object $licenses, $start, array $data)
    {
        $parentIndex = $start;

        foreach ($licenses as $key => $license) {
            $parentIndex++;
            $nestedData = $this->getNestedData($license, $parentIndex);

            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Formatting Software Licenses for listing page
     * 
     * @param object $license
     * @param int $index
     * 
     * @return array 
     */
    public function getNestedData(object $license, int $index)
    {
        $nestedData['id']         = $index;
        $nestedData['category']   = optional($license->category)->name;
        $nestedData['manufacturer']     = optional($license->softwareLicenseManufacturer)->name;
        $nestedData['vendor']     = optional($license->vendor)->name;

        if (auth()->user()->can('Software Assets Manage')) {
            $nestedData['name'] = "<a href='/software-assets/" . $license->id . "' style='margin-right: 10px'>" . $license->name . "</a>";
            $nestedData['action'] = "<a href='/software-assets/" . $license->id . "/edit' class='btn btn-link'><i class='icon icon-n-edit'></i></a>
            <a class='btn btn-link delete-license' data-id='" . $license->id . "' data-toggle='modal' data-target='#deleteLicenseModal'><i class='icon icon-delete-forever'></i></a>";
        } else {
            $nestedData['name'] = $license->name;
        }
        $nestedData['order_number'] = $license->order_number;
        $nestedData['type'] = $license->type ? config('software-license.types.' . $license->type) : '';
        $nestedData['license_key_type'] = $license->license_key_type ? config('software-license.license_key_types.' . $license->license_key_type) : '';
        $nestedData['key_usage_type'] = $license->key_usage_type ? config('software-license.key_usage_types.' . $license->key_usage_type) : '';
        $nestedData['license_type'] = $license->license_type ? config('software-license.license_types.' . $license->license_type) : '';
        $nestedData['renewal_date'] = $license->renewal_date;
        $nestedData['licenses'] = in_array($license->license_key_type, ['user_license']) ? $license->licenses_purchased : 'N/A';
        $nestedData['licenses_used'] = $license->licenses_used;
        $nestedData['keys_added'] = in_array($license->license_key_type, ['user_license']) ? $license->license_keys_added : 'N/A';

        return $nestedData;
    }

    /**
     * get Software License data
     * 
     * @param int $id
     * 
     * @return object
     */
    public function getLicense(int $id)
    {
        return $this->licenseRepo->getLicense($id);
    }

    /**
     * Update Software License
     * 
     * @param array $data
     * 
     * @return bool 
     */
    public function updateLicense($data)
    {
        DB::beginTransaction();
        try {
            $data = $this->formatLicenseData($data);
            $oldData = $this->licenseRepo->getLicense($data['id']);
            if ($data['license_key_type'] == 'no_license') {
                $this->licenseRepo->deleteLicenseKeys($data['id']);
                $data['license_keys_added'] = 0;
                $data['licenses_purchased'] = 0;
            } else if ($data['license_key_type'] == 'single_license') {
                $data['licenses_purchased'] = 0;
            } else {
                $updateData = ['key_usage_type' => $data['key_usage_type']];
                if ($data['key_usage_type'] == 'unlimited_usage') {
                    $updateData['users_allowed_for_key'] = 'U';
                    $updateData['no_of_users'] = 'U';
                }
                $this->licenseRepo->updateLicenseKeys($data['id'], $updateData);
            }
            $this->licenseRepo->updateLicense($data);
            $this->addLicenseHistoryForUpdate($oldData, $data);
            DB::commit();
            return true;
        } catch (Exception $e) {
            DB::rollback();
            Log::error($e->getMessage());
            // dd($e->getMessage(),$e->getLine(),$e->getFile());
            return false;
        }
    }
    /**
     * add License History For create
     */
    public function addLicenseHistoryForCreate($asset)
    {
        $description = $this->licenseHistoryService->getStatusDescriptionForCreated($asset);
        $this->licenseRepo->addLicenseHistory([
            'software_license_id'   => $asset->id,
            'action'                => 'created',
            'description'           => $description,
            'user_id'               => Auth::id()
        ]);
    }
    /**
     * add License History For Upadte
     */
    public function addLicenseHistoryForUpdate($oldData, $data)
    {
        $changedData = [];
        $historyData = [];
        $historyData['software_license_id'] = $data['id'];
        $historyData['action'] = 'updated';
        foreach ($data as $key => $value) {
            if (($value || $value == 0) || (isset($oldData->key) && $oldData->key)) {
                if ($value != $oldData->getRawOriginal($key)) {
                    $changedData[$key] = [
                        'old' => $oldData->getRawOriginal($key),
                        'new' => $value
                    ];

                    $historyData['old_' . $key] = $oldData->getRawOriginal($key);
                    $historyData['new_' . $key] = $value;
                }
            }
        }
        $description = $this->licenseHistoryService->getStatusDescriptionForEdited($oldData, $changedData);

        $historyData['description'] = $description;
        $historyData['user_id'] = Auth::id();
        $this->licenseRepo->addLicenseHistory($historyData);
    }

    /**
     * formatting license data before insert and update
     * 
     * @param array $data
     * 
     * @return array
     */
    public function formatLicenseData($data)
    {
        $data['renewal_date'] = null;
        if (in_array($data['license_type'], ['Subscription', 'subscription'])) {
            $data['renewal_date'] = (isset($data['subscription_expiration']) && $data['subscription_expiration']) ? convert_to_db_date($data['subscription_expiration']) : $this->getRenewalDate($data['start_date'], $data['software_license_subscription_id'], $data['license_type']);
        }
        $data['start_date'] = convert_to_db_date($data['start_date']);
        $data['licenses_purchased'] = $data['licenses_purchased'] ?? 0;
        $data['auto_renewal'] = $data['auto_renewal'] ?? 'n/a';
        if ($data['license_key_type'] == 'no_license') {
            $data['key_usage_type'] = null;
        }
        unset($data['subscription_expiration']);

        return $data;
    }

    /**
     * Delete Software License data
     * 
     * @param int $id
     * 
     * @return object
     */
    public function deleteLicense($id)
    {
        try {
            $this->licenseRepo->turnOffIntegrationStatus($id);  // turn off integration status in API credentials
            $this->licenseRepo->deleteLicenseUsers($id);
            $this->licenseRepo->deleteIntegrationUsers($id);
            $this->licenseRepo->deleteLicenseDocuments($id);
            $this->licenseRepo->deleteLicense($id);
            $this->licenseRepo->deleteLicenseHistories($id);
            return true;
        } catch (Exception $e) {
            Log::error($e->getMessage());

            return false;
        }
    }

    /**
     * It returns an array that containing the month name and the number of users added to the license in that month 
     * @param softwareAssetId The id of the software asset you want to get the data for.
     */
    public function getUserMonthGraphData($softwareAssetId)
    {
        $start = now()->subMonths(11)->startOfMonth();
        $graphData = [];
        foreach (CarbonPeriod::create($start, '1 month', Carbon::today()) as $date) {
            $lastDay = $date->endOfMonth()->format('Y-m-d');
            $teqUsersCount = $this->teqUserService->getActiveTeqUsersCount($softwareAssetId, $lastDay);

            $nonTeqUsersCount = $this->nonTeqUserService->getActiveNonTeqUsersCount($softwareAssetId, $lastDay);

            $totalUsersCount = $teqUsersCount + $nonTeqUsersCount;

            $graphData[] = array(
                'month' => $date->monthName,
                'user_count' => $totalUsersCount
            );
        }
        return  $graphData;
    }
}
