<?php

namespace App\Services\SoftwareLicense\ImportedAssets\Zoom;

use App\Repositories\ApiCredentialRepository;
use App\Services\Integrations\SoftwareLicense\ZoomIntegration;
use App\Repositories\SoftwareLicense\ZoomDataRepository;
use App\Http\Traits\SoftwareLicense\SoftwareAssetTrait;
use App\Models\ZoomMeeting;
use App\Models\ZoomMember;
use App\Services\SoftwareLicense\ImportedAssets\ImportedAssetsServiceInterface;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ZoomService implements ImportedAssetsServiceInterface
{

    use SoftwareAssetTrait;

    //pre defined data added from API doc
    protected $assetApiUserTypes = [
        "basic" => 1,
        "licensed" => 2,
        "none" =>  99
    ];
    private $slug;
    private $name;
    private $apiCredentialRepo;
    private $integration;
    private $softwareAssetDataRepo;

    /**
     * Constructor for ZoomService
     * 
     * @param ApiCredentialRepository $apiCredentialRepo
     * @param ZoomIntegration $integration
     * @param ZoomDataRepository $softwareAssetDataRepo
     */
    public function __construct(ApiCredentialRepository $apiCredentialRepo, ZoomIntegration $integration, ZoomDataRepository $softwareAssetDataRepo)
    {
        $this->slug = 'zoom';
        $this->name = 'Zoom';
        $this->apiCredentialRepo = $apiCredentialRepo;
        $this->integration = $integration;
        $this->softwareAssetDataRepo = $softwareAssetDataRepo;
    }

    /**
     * get license api credentials
     * 
     * @return mixed
     */
    public function getCredentials()
    {
        $data = Cache::get('zoomCredentials');

        if ($data === null) {
            $data =  $this->apiCredentialRepo->getCredentials($this->slug);
            Cache::put('zoomCredentials', $data);
        }
        return $data;
    }

    /**
     * check status of the license api connection
     * 
     * @param object $credentialsData
     * 
     * @return bool
     */
    public function checkConnection($credentialsData)
    {
        $accessToken = $this->getAccessToken($credentialsData);

        return $this->integration->checkConnection($accessToken);
    }

    /**
     * fetch access token from license api credentials data
     * 
     * @param mixed $credentialsData
     * 
     * @return mixed
     */
    public function getAccessToken($credentialsData)
    {
        if ($credentialsData) {
            $tokenData = $credentialsData->data ? json_decode($credentialsData->data) : [];

            return isset($tokenData->access_token) ? $tokenData->access_token : '';
        }

        return false;
    }

    /**
     * add credentials for license api
     * 
     * @param array $credentialsData
     * 
     * @return bool
     */
    public function addCredentials($credentialsData)
    {
        try {
            $this->apiCredentialRepo->addCredentials($credentialsData, $this->slug, $this->name);

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->error($this->name . ' credential create : ' . $e->getMessage());

            return false;
        }
    }

    /**
     * update credentials for license api
     * 
     * @param array $data
     * 
     * @return bool
     */
    public function updateCredentials($data)
    {
        try {
            $credentialsData = $this->getCredentials();

            if (!$credentialsData) {
                return false;
            }
            $updateData = [
                'user_name' => $data['app_name'] ? $data['app_name'] : $credentialsData->user_name,
                'key'       => $data['app_key'] ? $data['app_key'] : $credentialsData->key,
                'password'  => $data['app_secret'] ? $data['app_secret'] : $credentialsData->password
            ];

            if ($data['app_key'] != '' || $data['app_secret'] != '') {
                $updateData['data'] = '';
            }
            $this->apiCredentialRepo->updateCredentials($updateData, $this->slug);
            Cache::forget('zoomCredentials');

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->error($this->name . ' credential create : ' . $e->getMessage());

            return false;
        }
    }

    /**
     * set connection for license api
     * 
     * @param array $data
     * 
     * @return bool
     */
    public function setConnection()
    {
        $credentialsData = $this->getCredentials();

        if (!$credentialsData) {
            return false;
        }
        $tokenData = $credentialsData->data ? json_decode($credentialsData->data, true) : [];
        $generatedAccessToken = $this->integration->generateAccessToken($credentialsData->user_name, $credentialsData->key, $credentialsData->password);

        if ($generatedAccessToken) {
            $tokenData['access_token'] = $generatedAccessToken;
            $dataToUpdate = ['data' => json_encode($tokenData)];
            $this->apiCredentialRepo->updateCredentials($dataToUpdate, $this->slug);
            Cache::forget('zoomCredentials');
        }
        $accessToken = $tokenData['access_token'] ?? '';

        return $this->integration->checkConnection($accessToken);
    }

    /**
     * This function changes the integration status of license API and enables/disables the license software asset accordingly
     * 
     * @return bool
     */
    public function integrationStatusChange()
    {
        $integrationStatus = request('integration_status') ?? NULL;

        try {
            $dataToUpdate = ['integration_status' => $integrationStatus];
            $this->apiCredentialRepo->updateCredentials($dataToUpdate, $this->slug);
            Cache::forget('zoomCredentials');

            $licensedata = $this->getLicenseData();

            if ($licensedata) {
                $data = ['status' => $integrationStatus];
                $statusUpdate = $this->softwareAssetDataRepo->updateLicenseData($licensedata->id, $data);

                if ($statusUpdate && ($licensedata->status != $integrationStatus)) {
                    $data['id'] = $licensedata->id;
                    $updateLicenseHistoryData = $this->setImportedAssetHistoryForUpdateWithIntegration($licensedata, $data);
                    $this->softwareAssetDataRepo->addLicenseHistory($updateLicenseHistoryData);
                }
            } else {
                $softwareAssetData = $this->formatAssetDataForCreateWithIntegration($this->name);
                $assetCreated = $this->softwareAssetDataRepo->createLicenseData($softwareAssetData);
                $addLicenseHistoryData = $this->setImportedAssetHistoryForCreateWithIntegration($assetCreated);
                $this->softwareAssetDataRepo->addLicenseHistory($addLicenseHistoryData);
            }

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->error($this->name . ' integration status update error : ' . $e->getMessage());

            return false;
        }
    }

    /**
     * This function fetch license software asset data
     * 
     * @return mixed
     */
    public function getLicenseData()
    {
        return $this->softwareAssetDataRepo->getLicenseData();
    }

    /**
     * This function updates the number of licenses purchased for a license  asset
     * 
     * @param object $assetDetails
     * 
     * @return bool 
     */
    public function updateLicencePurchasedCount($credentialsData, $assetDetail)
    {
        try {
            $accessToken = $this->getAccessToken($credentialsData);
            $licensePurchasedCount = $this->integration->updateLicencePurchasedCount($accessToken);
            $data = ['licenses_purchased' => $licensePurchasedCount];
            $this->softwareAssetDataRepo->updateLicenseData($assetDetail->id, $data);

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->error($this->name . ' update license purchased count update error : ' . $e->getMessage());

            return false;
        }
    }

    /**
     * This function creates license members by fetching a list of members from API and add to license  members table.
     * 
     * @param mixed $credentialsData
     * 
     */
    public function createMembers($credentialsData)
    {
        $nextPageToken = '';

        try {
            $accessToken = $this->getAccessToken($credentialsData);
            $this->softwareAssetDataRepo->clearExistingDataBeforeFetching();
            $rolesList = $this->getRolesList($credentialsData);
            $memberDataForCreate = [];

            do {
                $softwareAssetMemberList = $this->integration->getSoftwareAssetMemberList($accessToken, $nextPageToken);
                $nextPageToken  = $softwareAssetMemberList['next_page_token'];

                if (isset($softwareAssetMemberList['users'])) {

                    foreach ($softwareAssetMemberList['users'] as $member) {

                        if ($member['type'] == $this->assetApiUserTypes["licensed"]) {
                            $memberData = $this->getMemberData($member, $rolesList);
                            $memberDataForCreate[] = $memberData;
                        }
                    }
                }
            } while ($nextPageToken);

            if (!empty($memberDataForCreate)) {
                $this->softwareAssetDataRepo->createMembers($memberDataForCreate);
            }

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->info($this->name . " team members fetch error ---- " . $e->getMessage());

            return false;
        }
    }

    /** 
     * It parses the response from the license  API and returns an array of data to create or update software asset members. 
     * @param member The member object returned by the license API
     * @param array $rolesList Roles of Software License
     * 
     * @return array
     */
    public function getMemberData($member, $rolesList)
    {
        $user = $this->getUserForLicenseMemeber('email', $member['email']);
        $data['user_id']             = $user ? optional($user)->id : null;
        $data['zoom_user_id']        = $member['id'];
        $data['email']               = $member['email'];
        $data['email_verified']      = $member['verified'];
        $data['status']              = $member['status'];
        $data['name']                = $member['display_name'];
        $data['joined_on']           = $member['user_created_at'];
        $data['role']                = isset($rolesList[$member['role_id']]) ? $rolesList[$member['role_id']]['name'] : '';

        return $data;
    }

    /**
     * Fetch roles of the license api with API
     * 
     * @param object $credentialsData
     * 
     * @return array
     */
    public function getRolesList($credentialsData)
    {
        $accessToken = $this->getAccessToken($credentialsData);
        $rolesApiResponse = $this->integration->getRolesList($accessToken);
        $roleData = [];

        foreach ($rolesApiResponse['roles'] as $singleRole) {
            $roleData[$singleRole['id']] = ['name' => $singleRole['name']];
        }

        return $roleData;
    }

    public function softwareAssetMemberListSync($assetDetails)
    {
        try {
            $softwareAssetMembers = $this->softwareAssetDataRepo->getMemberData();
            $usersToBeDeleted = $this->softwareAssetDataRepo->getOldUsersToBeDeleted($assetDetails->id);
            $this->softwareAssetDataRepo->deleteOldUsersToBeDeleted($assetDetails->id, $usersToBeDeleted);
            $softwareUserhistorydata = $this->setSoftwareUserHistoryData('user_removed', $assetDetails, $usersToBeDeleted);
            $this->softwareAssetDataRepo->addLicenseHistory($softwareUserhistorydata);

            if (!$softwareAssetMembers) {
                return true;
            }

            $newlyAddedUser = [];

            foreach ($softwareAssetMembers as $softwareAssetMemberData) {
                $licenceUserData = $this->getSoftwareAssetUserData($assetDetails, $softwareAssetMemberData);
                $conditions = [
                    'software_license_id' => $assetDetails->id,
                    'user_id'   => $softwareAssetMemberData->user_id
                ];
                $update = $this->softwareAssetDataRepo->updateLicenseUsers($conditions, $licenceUserData);


                if ($update->wasRecentlyCreated) {
                    $newlyAddedUser[] = $softwareAssetMemberData->user_id;
                }
            }

            if (!empty($newlyAddedUser)) {
                $softwareUserhistorydata = $this->setSoftwareUserHistorydata('user_added', $assetDetails, $newlyAddedUser);
                $this->softwareAssetDataRepo->addLicenseHistory($softwareUserhistorydata);
            }

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->info($this->name . " member list sync ---- " . $e->getMessage());

            return false;
        }
    }

    /**
     * Updates the owner of a software asset
     * 
     * @param mixed $assetDetails
     * 
     * @return bool
     */
    public function updateSoftwareAssetOwner($assetDetails)
    {
        try {
            $owner = $this->softwareAssetDataRepo->getSoftwareLicenseOwner('Owner');

            if ($owner) {
                $data = ['owner_id' => $owner->user_id];
                $this->softwareAssetDataRepo->updateLicenseData($assetDetails->id, $data);
            }

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->info("Update " . $this->name . " owner ---- " . $e->getMessage());

            return false;
        }
    }

    /**
     * Updates the licenses used count of a software asset with the count of all active users in asset members
     * 
     * @param object $assetDetails
     * 
     * @return bool
     */
    public function updateLicensesUsedCount($assetDetails)
    {
        try {
            $count = $this->softwareAssetDataRepo->getSoftwareLicenseActiveUserCount();

            if ($count > 0) {
                $data = ['licenses_used' =>  $count];
                $this->softwareAssetDataRepo->updateLicenseData($assetDetails->id, $data);
            }

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->info("Update " . $this->name . " licenses used count ---- " . $e->getMessage());

            return false;
        }
    }

    /**
     * Syncs the list of Zoom meetings.
     *
     * @param mixed $fromDate The start date for syncing.
     * @param mixed $toDate   The end date for syncing.
     * @throws Exception     If the data import fails.
     * @return bool          True if the sync is successful, false otherwise.
     */
    public function zoomMeetingsListSync($fromDate, $toDate)
    {
        try {
            $zoomMembers = ZoomMember::get();
            $credentialsData = $this->getCredentials();
            $accessToken = $this->getAccessToken($credentialsData);

            foreach ($zoomMembers as $zoomMember) {

                $response  =  $this->integration->getMemberHostedMeetings($accessToken, $zoomMember, $fromDate, $toDate);

                $this->storeMeetingsData($response['meetings']);
                // Continue making API calls until there's no next page
                while (!empty($response['next_page_token'])) {
                    // Make the next API call with the updated page_token
                    $response  =  $this->integration->getMemberHostedMeetings($accessToken, $zoomMember, $fromDate, $toDate, $response['next_page_token']);
                    // Process the current response
                    $this->storeMeetingsData($response['meetings']);
                }
            }

            return true;
        } catch (Exception $e) {
            Log::channel('daily')->info("Update " . $this->name . " meetings data import failed " . $e->getMessage() . ' - ' . $e->getLine() . ' - ' . $e->getFile());

            return false;
        }
    }

    /**
     * Stores the Zoom meeting data in the database.
     *
     * @param array $zoomMeetingData The Zoom meeting data to be stored.
     * @return bool Returns true if the Zoom meeting data is stored successfully, otherwise false.
     */
    private function storeMeetingsData($zoomMeetingData)
    {
        if (empty($zoomMeetingData)) {
            return true;
        }


        foreach ($zoomMeetingData as $meeting) {
            // Create an array representing the Zoom meeting data
            $meetingData = [
                'zoom_meeting_id'    => $meeting['id'],
                'host_id'            => $meeting['host_id'],
                'topic'              => $meeting['topic'],
                'type'               => $meeting['type'],
                'start_time'         => $meeting['start_time'],
                'duration'           => $meeting['duration'],
                'timezone'           => $meeting['timezone'],
                'meeting_created_at' => $meeting['created_at'],
                'join_url'           => $meeting['join_url'],
            ];

            ZoomMeeting::firstOrCreate(['zoom_meeting_id' => $meeting['id']], $meetingData);
        }

        return true;
    }
}
