<?php

namespace App\Services\SoftwareLicense\ImportedAssets\Zoom;

use App\Models\ZoomMeeting;
use App\Http\Responses\ReportOutputData;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ZoomMeetingService
{

    /**
     * Constructs a new instance of the class.
     *
     * @param ReportOutputData $reportOutputData The ReportOutputData object.
     */
    public function __construct(protected  ReportOutputData $reportOutputData)
    {
    }

    /**
     * Retrieves data based on the provided host ID.
     *
     * @param int $hostId The ID of the host.
     * @return array The data containing meetings and count.
     */
    public function data($hostId)
    {

        $inputData  = $this->getInputData(request('form'));

        $meetings = ZoomMeeting::select(
            DB::raw("DATE_FORMAT(start_time, '%m/%Y') as month_year"),
            'host_id',
            DB::raw('COUNT(*) as meeting_count'),
            DB::raw('AVG(duration) as avg_duration')
        )->with('host:id,email,zoom_user_id')
            ->where('host_id', request('host_id'))
            ->whereBetween('start_time', [Carbon::now()->subYear(), now()])
            ->groupBy(DB::raw("DATE_FORMAT(start_time, '%m/%Y')"));

        $meetings     = $this->filter($meetings, $inputData);
        $count      = $meetings->count();
        $meetings = $this->reportOutputData->getOutputData($meetings);

        return compact('meetings', 'count');
    }

    /**
     * Retrieves the input data for the given input.
     *
     * @param mixed $input The input parameter to retrieve data for.
     * @return array The input data as an array.
     */
    private function getInputData($input)
    {
        return [];
    }

    /**
     * Filter the meetings based on the given input data.
     *
     * @param array $meetings The array of meetings to be filtered.
     * @param mixed $inputData The input data to filter the meetings.
     * @return mixed The filtered array of meetings.
     */
    private function filter($meetings, $inputData)
    {
        return $meetings;
    }

    /**
     * Retrieves the month-wise user meetings data.
     *
     * @param array $meetings The array of meetings.
     * @param int $start The starting index.
     * @param array $data The existing data array.
     * @return array The updated data array.
     */
    public function getMonthWiseUserMeetingsData($meetings, $start, $data)
    {
        $parentIndex = $start;

        foreach ($meetings as $key => $meeting) {

            $parentIndex++;

            $nestedData = $this->getNestedMonthWiseUserMeetingsData($meeting, $parentIndex);

            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Generates the nested month-wise user meetings data.
     *
     * @param mixed $meeting The meeting data.
     * @param int $index The index of the meeting.
     * @return array The nested data containing the meeting details.
     */
    public function getNestedMonthWiseUserMeetingsData($meeting, $index)
    {
        $nestedData['id']           = $index;
        $nestedData['user_email']    = optional($meeting->host)->email;
        $nestedData['month']     = $meeting->month_year;
        $nestedData['meetings_count']         = $meeting->meeting_count;
        $nestedData['average_duration']      = round($meeting->avg_duration);
        return $nestedData;
    }
}
