<?php

namespace App\Services\Slack;

use App\Models\ApiCredential;
use App\Models\SlackNotificationCredential;
use App\Services\Integrations\Slack\SlackApiIntegration;
use Exception;
use Illuminate\Support\Facades\Log;

class SlackService
{

    public function __construct(protected SlackApiIntegration $slackApiService) {}

    /**
     * Checks the connection to the Slack API.
     *
     * @return bool Returns true if the connection is successful, false otherwise.
     */
    public function checkSlackConnection()
    {
        $connection = ApiCredential::where('slug', 'slack_api')->first();
        return optional($connection)->key ? true : false;
    }

    /**
     * Store the access token in the database.
     *
     * @param array $accessDetails The access details.
     * @return int The number of rows affected.
     */
    public function storeAccessToken($accessDetails)
    {
        return ApiCredential::where('slug', 'slack_api')->first()->update([
            'user_name' => $accessDetails['bot_user_id'] ?? null,
            'key' => $accessDetails['access_token'] ?? null,
            'password' => $accessDetails['authed_user']['access_token'] ?? null,
        ]);
    }

    /**
     * Filters the channels based on the given search string.
     * 
     * @param string $search The search string to filter the channels.
     * @throws Exception If an error occurs while filtering the channels.
     * @return mixed|null Returns the first channel that matches the search string, or null if no match found.
     */
    public function getChannelDetails($search)
    {
        $searchedChannel = [];

        try {
            $availableChannels = $this->slackApiService->getAllChannels();

            $searchedChannel = $this->findChannels($availableChannels, $search);

            if (!empty($searchedChannel) === true) {
                return $searchedChannel;
            }

            return $this->searchChannelInPaginatedResults($search, $availableChannels);
        } catch (Exception $e) {
            Log::channel('daily')->error($e->getMessage() . ' ' . $e->getFile() . ' ' . $e->getLine());
            return null;
        }

        return $searchedChannel;
    }

    /**
     * Find a channel in the given array of available channels by name.
     *
     * @param array $availableChannels The array of available channels.
     * @param string $search The name of the channel to search for.
     * @return mixed|null Returns the channel object if found, null otherwise.
     */
    private function findChannels($availableChannels, $search)
    {
        if (is_array($availableChannels) === false) {
            Log::channel('daily')->error("Failed to fetch channel $search details.");

            return null;
        }

        if (isset($availableChannels['channels']) === true && !empty($availableChannels['channels']) === true) {
            $channelsCollection = collect($availableChannels['channels']);
            return $channelsCollection->firstWhere('name', $search);
        }

        return null;
    }

    /**
     * Search for a channel in paginated results.
     *
     * @param string $search The name of the channel to search for.
     * @param array $availableChannels The available channels to search in.
     * @return array|null The searched channel, or null if not found.
     */
    private function searchChannelInPaginatedResults($search,  $availableChannels)
    {
        $nextCursor = $availableChannels['response_metadata']['next_cursor'] ?? '';

        while ($nextCursor !== '') {

            $result = $this->slackApiService->getAllChannels($nextCursor);

            $searchedChannel = $this->findChannels($result, $search);

            if (!empty($searchedChannel) === true) {
                return $searchedChannel;
            }

            $nextCursor = $result['response_metadata']['next_cursor'] ?? '';
        }

        return null;
    }

    /**
     * Retrieves the channel information by its ID.
     *
     * @param string $channelId The ID of the channel.
     * @throws Exception If an error occurs while retrieving the channel information.
     * @return array|null The channel information, or null if not found.
     */
    public function getChannelInfoById($channelId)
    {
        try {
            $channelInfo = $this->slackApiService->getChannelInfoById($channelId);

            if (!empty($channelInfo) === true) {
                return $channelInfo;
            }

            return null;
        } catch (Exception $e) {
            Log::channel('daily')->error($e->getMessage() . ' ' . $e->getFile() . ' ' . $e->getLine());
            return null;
        }
    }

    /**
     * Sends a notification to a specified channel.
     *
     * @param int $channel The channel to send the notification to.
     * @param string|null $message The message to send.
     * @return bool Returns true if the notification was sent successfully, false otherwise.
     */
    public function sendMessageToChannel($channel, $message = null)
    {
        $channelDetails = SlackNotificationCredential::where('slug', $channel)->first();

        if ($channelDetails && $channelDetails->channel_id && $channelDetails->status && $message) {
            return $this->slackApiService->sendMessageToChannel($channelDetails->channel_id, $message);
        }

        return false;
    }

    /**
     * Sends a message to a user through a specified channel.
     *
     * @param string $email The email of the user to send the message to.
     * @param string|null $message The message to send to the user. Default is null.
     * @param string|null $channel The channel to send the message through. Default is null.
     * @param  $attachments The attachments to send , can be interactive buttons.
     * @throws Exception if channel details are not found or if the user is not found.
     * @return bool Returns true if the message is successfully sent, false otherwise.
     */
    public function sendMessageToUser($email, $message = null, $channel = null, $attachments = null)
    {
        $channelDetails = SlackNotificationCredential::where('slug', $channel)->first();

        if ($channelDetails && $channelDetails->status && $email && $message !== null) {
            $userDetails = $this->slackApiService->searchUser($email);
            if ($userDetails && isset($userDetails['user'])) {
                $userId = $userDetails['user']['id'];
                return $this->slackApiService->sendMessageToChannel($userId, $message, $attachments);
            }
            Log::channel('daily')->error($email . " - User not found");
        }

        return false;
    }

    /**
     * Updates a Slack message in the specified channel with the given message, attachments, and timestamp.
     *
     * @param string $channelId The ID of the channel where the message is located.
     * @param string $message The new message to update the Slack message with.
     * @param mixed|null $attachments The attachments to update the Slack message with. Default is null.
     * @param string|null $messageTs The timestamp of the message to update. Default is null.
     * @return bool
     */
    public function updateSlackMessage($channelId, $message, $attachments = null, $messageTs = null)
    {
        return $this->slackApiService->updateSlackMessage($channelId, $message, $attachments, $messageTs);
    }
}
