<?php

namespace App\Services\Slack;

use App\Services\Integrations\Slack\SlackApiIntegration;
use App\User;
use Illuminate\Support\Facades\URL;

class SlackMessageHelperService
{

    public function __construct(protected SlackApiIntegration $slackApiService) {}

    /**
     * Generates a message for loaner assets that are overdue as of yesterday's date.
     *
     * @param array $assets An array of loaner assets.
     * @return string The generated message.
     */
    public function getOverDueLoanerDateUserMessage($assets)
    {
        $url = URL::to('/');
        $message = parse_date_from_db_datetime(\Carbon\Carbon::now()) .  PHP_EOL . PHP_EOL;
        $message .= optional($assets[0]->user)->userName . "," . PHP_EOL . PHP_EOL;

        $message .= '*The expected return date for the following assets has expired:*' . PHP_EOL . PHP_EOL;

        foreach ($assets as $asset) {
            $message .= "*<" . $url . "/assets/" . $asset->id . "|" . $asset->serial_no . ">* - " . optional($asset->makeAndModel)->makeModelName . PHP_EOL;
            $message .= "  - Loaner Item Due Date: " . $asset->loaner_return_date . PHP_EOL . PHP_EOL;
        }

        $message .= "Please return the items to the Service Desk promptly." . PHP_EOL . PHP_EOL;

        $message .= PHP_EOL . 'Thank You,' . PHP_EOL . 'IT Support';

        return $message;
    }

    /**
     * Generates a message for loaner assets with overdue return dates.
     *
     * @param array $assets The array of loaner assets.
     * @return string The generated message.
     */
    public function getOverDueLoanerDateAdminMessage($assets)
    {
        $url = URL::to('/');
        $message = '';
        $commonUserToNotify = $this->slackApiService->searchUser(config('notification.slack.common_user_to_notify', ''));

        if ($commonUserToNotify) {
            $slackCommonUserId = $commonUserToNotify['user']['id'];
            $message .=  "<@$slackCommonUserId>" . PHP_EOL . PHP_EOL;
        }

        $message .= parse_date_from_db_datetime(\Carbon\Carbon::now()) .  PHP_EOL . PHP_EOL;
        $message .= '*The expected return date for the following assets has expired:*' . PHP_EOL . PHP_EOL;

        foreach ($assets as $asset) {
            $message .= "*<" . $url . "/assets/" . $asset->id . "|" . $asset->serial_no . ">* - "  . optional($asset->makeAndModel)->makeModelName .  PHP_EOL;
            $message .= "  - User: " . optional($asset->user)->userName  . " - " . optional($asset->user)->email . PHP_EOL;
            $message .= "  - Expected Return Date: " . $asset->loaner_return_date . PHP_EOL . PHP_EOL;
        }

        return $message;
    }

    /**
     * Generates a message for loaner assets with return dates due in the next 7 days.
     *
     * @param array $assets The array of loaner assets.
     * @return string The generated message.
     */
    public function getLoanerDueNextWeekUserMessage($assets)
    {
        $url = URL::to('/');
        $message = parse_date_from_db_datetime(\Carbon\Carbon::now()) .  PHP_EOL . PHP_EOL;
        $message .= optional($assets[0]->user)->userName . "," . PHP_EOL . PHP_EOL;

        $message .= '*The expected return date for the following loaned assets is approaching in 7 days:*' . PHP_EOL . PHP_EOL;

        foreach ($assets as $asset) {
            $message .= "*<" . $url . "/assets/" . $asset->id . "|" . $asset->serial_no . ">* - " . optional($asset->makeAndModel)->makeModelName . PHP_EOL;
            $message .= "  - Expected Return Date: " . $asset->loaner_return_date . PHP_EOL . PHP_EOL;
        }

        $message .= "Please return the items to the Service Desk promptly." . PHP_EOL . PHP_EOL;

        $message .= PHP_EOL . 'Thank You,' . PHP_EOL . 'IT Support';

        return $message;
    }
}
