<?php

namespace App\Services\Slack;

use App\Http\Responses\ReportOutputData;
use App\Models\SlackNotificationCredential;
use App\Repositories\CommonFilterRepository;

class SlackChannelsDataService
{

    /**
     * Constructor for class
     *
     * @param CommonFilterRepository $commonFilterRepository The common filter repository
     * @param ReportOutputData $reportOutputData The report output data
     */
    public function __construct(
        protected CommonFilterRepository $commonFilterRepository,
        protected ReportOutputData $reportOutputData,
    ) {}


    /**
     * Gets the data for the Slack channels data table.
     *
     * @return array Data for the Slack channels data table.
     */
    public function data()
    {
        $channels = new SlackNotificationCredential;
        $count  = $channels->count();
        $channels = $this->reportOutputData->getOutputData($channels, ['name' => 'asc']);

        return compact('channels', 'count');
    }

    /**
     * Generating data table data from qurey result
     * @param $query Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     *
     * @return array
     */
    public function getReportData($query, $start, $data = [])
    {
        $parentIndex = $start;

        foreach ($query as $singleRecord) {
            $parentIndex++;
            $nestedData = $this->getNestedData($singleRecord, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Return an array of nested data for the given slack notification configuration.
     *
     * @param \App\Models\SlackNotificationCredential $configurationData The slack notification configuration
     * @param int $index The index of the record
     *
     * @return array The nested data array
     */
    public function getNestedData($configurationData, $index)
    {
        return [
            'id' => $index,
            'name' => $configurationData->name,
            'channel_name' => $configurationData->channel_name,
            'connection' => $this->getConnectionStatus($configurationData),
            'toggle_switch' => $this->getToggleSwitch($configurationData),
            'edit_button' => $this->getEditButton($configurationData),
            'test_button' => $this->getTestButton($configurationData),
        ];
    }


    /**
     * Return the connection status for the given slack notification configuration.
     *
     * If the channel name is 'Direct Message', return ' -- '.
     * If the channel name is not 'Direct Message', return 'Connected' if the channel_id is not null, otherwise
     * return 'Not Connected'.
     *
     * @param \App\Models\SlackNotificationCredential $configurationData The slack notification configuration
     *
     * @return string The connection status
     */
    private function getConnectionStatus($configurationData)
    {
        if ($configurationData->channel_name === 'Direct Message') {
            return ' -- ';
        }

        return $configurationData->channel_id ? 'Connected' : 'Not Connected';
    }

    /**
     * Return the toggle switch for the given slack notification configuration.
     *
     * If the user does not have the permission to edit API credentials, return an empty string.
     * Otherwise, return a toggle switch element with the checked status based on the configuration's status.
     *
     * @param \App\Models\SlackNotificationCredential $configurationData The slack notification configuration
     *
     * @return string The toggle switch element
     */
    private function getToggleSwitch($configurationData)
    {
        if (!auth()->user()->can('API Credentials Edit')) {
            return '';
        }

        return '
        <div class="p-10">
            <label class="toggle-switch">
                <input type="checkbox" class="messagingToggleButton" data-notification-id="' . $configurationData->id . '" ' . ($configurationData->status == 1 ? 'checked' : '') . '>
                <span class="toggle-slider"></span>
            </label>
        </div>';
    }

    /**
     * Return the test button element for the given slack notification configuration.
     *
     * If the channel name is 'Direct Message', return an empty string.
     *
     * @param \App\Models\SlackNotificationCredential $configurationData The slack notification configuration
     *
     * @return string The test button element
     */
    private function getTestButton($configurationData)
    {
        if ($configurationData->channel_name === 'Direct Message') {
            return '';
        }

        $disabled = (!$configurationData->channel_id || $configurationData->status == 0) ? 'disabled style="pointer-events:none;"' : '';
        $connectedStatus = !$configurationData->channel_id ? 'not-connected' : 'connected';

        return '
            <a class="btn btn-primary" id="test-notification-btn-' . $configurationData->id . '" 
               href="' . route('slack-notifications-config.test', ['slack_notifications_config' => $configurationData->id]) . '" ' . $disabled . '
               data-connected-status="' . $connectedStatus . '">Test</a>';
    }

    /**
     * Return the edit button element for the given slack notification configuration.
     *
     * If the user does not have the API Credentials Edit permission or the channel name is 'Direct Message', return an empty string.
     *
     * @param \App\Models\SlackNotificationCredential $configurationData The slack notification configuration
     *
     * @return string The edit button element
     */
    private function getEditButton($configurationData)
    {
        if (auth()->user()->can('API Credentials Edit') && $configurationData->channel_name !== 'Direct Message') {
            return '<a class="btn btn-primary" href="' . route('slack-notifications-config.edit', ['slack_notifications_config' => $configurationData->id]) . '">Edit</a>';
        }
        return '';
    }
}
