<?php

namespace App\Services\Settings;

use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;

/**
 * Service Class for Manufacturer Export
 */
class TechnicalSpecsService
{

    /**
     * Retrieve all technical specifications along with related make and model, asset type, manufacturer, and asset count.
     *
     * @return \Illuminate\Database\Eloquent\Builder The query builder with the specified relationships and counts.
     */
    public function getAllTechnicalSpecs()
    {
        return TechnicalSpecs::with(['makeAndModel:id,asset_type_id,name,manufacturer_id', 'makeAndModel.assetType:id,name', 'makeAndModel.manufacturer:id,name'])->withCount('assets')
            ->join('make_and_models', 'technical_specs.make_and_model_id', '=', 'make_and_models.id')
            ->leftJoin('asset_types', 'make_and_models.asset_type_id', '=', 'asset_types.id')
            ->leftJoin('manufacturers', 'make_and_models.manufacturer_id', '=', 'manufacturers.id');
    }

    /**
     * Filter the technical specifications based on asset type, manufacturer, make and model, and technical specification from request parameters.
     *
     * @param \Illuminate\Database\Eloquent\Builder $techSpecs The query builder for technical specifications.
     *
     * @return \Illuminate\Database\Eloquent\Builder The filtered query builder.
     */
    public  function filterTechnicalSpecs($techSpecs)
    {
        $assetType = request('asset_type');
        $manufacturer = request('manufacturer');
        $make_and_model = request('make_and_model');
        $technical_spec = request('technical_spec');

        if ($assetType) {
            $assetIds = MakeAndModel::whereIn('asset_type_id', $assetType)->pluck('id');
            $techSpecs = $techSpecs->whereIn('technical_specs.make_and_model_id', $assetIds);
        }

        if ($manufacturer) {
            $manufacturerIds = MakeAndModel::whereIn('manufacturer_id', $manufacturer)->pluck('id');
            $techSpecs = $techSpecs->whereIn('technical_specs.make_and_model_id', $manufacturerIds);
        }

        if ($make_and_model) {
            $techSpecs = $techSpecs->whereIn('technical_specs.make_and_model_id', $make_and_model);
        }

        if ($technical_spec) {
            $techSpecs = $techSpecs->whereIn('technical_specs.id', $technical_spec);
        }

        return $techSpecs;
    }

    /**
     * Retrieve technical specifications export data, starting from a given index, and appending nested data for each spec.
     *
     * @param array $specs The list of technical specifications to process.
     * @param int   $start The starting index.
     * @param array $data  The initial data array to which the nested data will be appended.
     *
     * @return array The updated data array with nested technical specifications data.
     */
    public function getTechnicalSpecsExportData($specs, $start, $data)
    {
        $parentIndex = $start;

        foreach ($specs as $spec) {
            $parentIndex++;
            $nestedData = $this->getExportNestedData($spec);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieve nested data for a given technical specification.
     *
     * @param object $spec The technical specification object containing details of the specification.
     *
     * @return array An associative array containing the asset type, manufacturer, hardware standard, technical specifications, and various values.
     */
    public function getExportNestedData($spec)
    {
        $nestedData['Asset Type']            =   optional(optional($spec->makeAndModel)->assetType)->name;
        $nestedData['Manufacturer']          =   optional(optional($spec->makeAndModel)->manufacturer)->name;
        $nestedData['Hardware Standard']     =   optional($spec->makeAndModel)->name;
        $nestedData['Tech Specs']            =   $spec->details;
        $nestedData['Original Value']        =   $spec->original_value ? $spec->original_value . ' USD' : '';
        $nestedData['Salvage Value']         =   $spec->new_value;
        $nestedData['Life of Asset (years)'] =   $spec->life_of_asset;
        $nestedData['# of Assets']           =   $spec->assets_count;

        return $nestedData;
    }
}
