<?php

namespace App\Services\Settings\History;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Settings\History\UserRoleChangeHistoryRepository;

class UserRoleChangeHistoryService
{
    /**
     * Constructor for RoleChangeHistoryDataService
     *
     * @param UserRoleChangeHistoryRepository $repository The role change history repository
     * @param CommonFilterRepository $commonFilterRepository The common filter repository
     * @param ReportOutputData $reportOutputData The report output data response
     */
    public function __construct(
        protected UserRoleChangeHistoryRepository $repository,
        protected CommonFilterRepository $commonFilterRepository,
        protected ReportOutputData $reportOutputData
    ) {}

    /**
     * Fetching role change history data
     *
     * @return array
     */
    public function data()
    {
        $histories = $this->repository->getRoleChangeHistory();
        $histories = $this->filter($histories);
        $count     = $histories->count();
        $histories = $this->reportOutputData->getOutputData($histories);

        return compact('histories', 'count');
    }

    /**
     * Filter RoleChangeHistory eloquent query
     *
     * @param $query
     *
     * @return eloquent::query()
     */
    private function filter($query)
    {
        $filterFields = $this->getInputData();
        $query        = $this->commonFilterRepository->filterWithDirectFields($query, 'user_id', $filterFields['user']);
        $query        = $this->commonFilterRepository->filterWithDirectFields($query, 'changed_by_user_id', $filterFields['changed_by']);
        $query        = $this->commonFilterRepository->filterWithDirectFields($query, 'from_user_type_id', $filterFields['from_role']);
        $query        = $this->commonFilterRepository->filterWithDirectFields($query, 'to_user_type_id', $filterFields['to_role']);
        $query        = $this->commonFilterRepository->filterWithDirectDateRange($query, 'created_at', $filterFields['date_from'], $filterFields['date_to']);

        return $query;
    }

    /**
     * Retrieves the input data from the request.
     *
     * This function retrieves the input data from the request, which is either obtained from the 'form' parameter

     * @return array The input data in the form of an associative array.
     */
    public function getInputData()
    {
        $requestData = request('form') ?? request()->all();

        return [
            'user'              => $requestData['user'] ?? '',
            'changed_by'        => $requestData['changed_by'] ?? '',
            'from_role'         => $requestData['from_role'] ?? '',
            'to_role'           => $requestData['to_role'] ?? '',
            'date_from'         => $requestData['date_from'] ?? '',
            'date_to'           => $requestData['date_to'] ?? ''
        ];
    }

    /**
     * Format whole history data into data table format array
     *
     * @param object $histories the histories collection of objects
     * @param int $start start index
     *
     * @return array $data
     */
    public function getData($histories, $start)
    {
        $data = [];

        foreach ($histories as $history) {
            $data[] = $this->getNestedData($history, ++$start);
        }

        return $data;
    }

    /**
     * Format a single history record into a data table format array
     *
     * @param $history the history object
     * @param int $index counter variable
     *
     * @return array
     */
    private function getNestedData($history, int $index): array
    {
        $data       = [];
        $nestedData = [];
        $nestedData = $this->getCommonNestedData($history);

        foreach ($nestedData as $key => $value) {
            $data[str_slug($key, '_')] = $value;
        }

        $data['id']         = $index;
        $data['user']       = generateUserLink($history->user_id, $history->user?->email);
        $data['changed_by'] = ($history->changedBy) ? generateUserLink($history->changed_by_user_id, $history->changedBy?->user_name) : $history->changed_by;

        return $data;
    }

    /**
     * Get data for export role change history
     *
     * @return object
     */
    public function getExportData()
    {
        $histories = $this->repository->getRoleChangeHistory();
        $histories = $this->filter($histories);

        return $histories->orderBy('created_at', 'DESC');
    }

    /**
     * Get data for export
     *
     * @param object $histories the histories collection of objects
     *
     * @return array $data
     */
    public function getReportExportData($histories)
    {
        $data = [];

        foreach ($histories as $history) {
            $data[] = $this->getCommonNestedData($history);
        }

        return $data;
    }

    /**
     * Returns an array containing common nested data from a given history object.
     *
     * @param object $history The history object to extract data from.
     *
     * @return array
     */
    private function getCommonNestedData($history): array
    {
        return [
            'User'       => optional($history->user)->email,
            'Changed By' => $history->changedBy->user_name ?? $history->changed_by,
            'From Role'  => optional($history->fromRole)->name,
            'To Role'    => optional($history->toRole)->name,
            'Date'       => $history->created_at,
        ];
    }

    /**
     * Adds a history for a user's role change from
     *
     * @param object $user The user object.
     *
     * @return bool
     */
    public function addRoleChangeHistory($user)
    {
        return $this->repository->addRoleChangeHistory($user);
    }
}
