<?php

namespace App\Services\Settings\History;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Settings\History\UserLoginHistoryRepository;

class UserLoginHistoryService
{
    /**
     * Constructor for the class.
     *
     * @param UserLoginHistoryRepository $repository
     * @param CommonFilterRepository $commonFilterRepository
     * @param ReportOutputData $reportOutputData
     */
    public function __construct(
        protected UserLoginHistoryRepository $repository,
        protected CommonFilterRepository $commonFilterRepository,
        protected ReportOutputData $reportOutputData
    ) {}

    /**
     * Data for user login history data table
     *
     * @return array
     */
    public function data()
    {
        $histories  = $this->repository->getHistory();
        $assets     = $this->filter($histories);
        $count      = $assets->count();
        $assets     = $this->reportOutputData->getOutputData($assets);

        return compact('assets', 'count');
    }

    /**
     * Filter the user login history records
     *
     * @param array $query
     *
     * @return App\Models\Assets
     */
    public function filter($query)
    {
        $inputData = $this->getInputData();
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'user_id', $inputData['user_id']);
        $query = $this->commonFilterRepository->filterWithDirectDateRange($query, 'created_at', $inputData['date_from'], $inputData['date_to']);

        return $query;
    }

    /**
     * Retrieves the user input data for filtering user login history records.
     *
     * @return array
     */
    private function getInputData()
    {
        $inputData = request('form') ?? request()->all();

        return [
            'user_id' => $inputData['user'] ?? '',
            'date_from' => $inputData['date_from'] ?? '',
            'date_to' => $inputData['date_to'] ?? '',
        ];
    }

    /**
     * making the out put for data table from query
     *
     * @param Collection $histories
     * @param int $start
     * @param array $data
     *
     * @return Array
     */
    public function getHistoryData($histories, $start, $data)
    {
        $parentIndex = $start;

        foreach ($histories as $history) {
            $parentIndex++;
            $nestedData = $this->getNestedData($history, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Creates an array with data table fields and their values
     *
     * @param $history
     * @param int $index
     *
     * @return array
     */
    public function getNestedData($history, int $index): array
    {
        return [
            'id'          => $index,
            'name'        => $history->user?->userName,
            'email'       => generateUserLink($history->user?->id, $history->user?->email),
            'title'       => $history->user?->business_title,
            'location'    => $history->user?->city,
            'system_role' => $history->user?->userType?->name,
            'status'      => $history->user?->status === 1 ? 'Active' : 'Inactive',
            'date'        => $history->created_at,
        ];
    }

    /**
     * Making the query for export functionality for the shipment condition report
     *
     * @return Collection
     */
    public function getExportData()
    {
        $histories  = $this->repository->getHistory();
        $histories  = $this->filter($histories);

        return $histories;
    }

    /**
     * Making Export data from the query result
     *
     * @param $assets  App\Models\AssetHistory collection
     *
     * @return Collection
     */
    public function getExportHistoryData($histories)
    {
        $data = [];

        foreach ($histories as $history) {
            $nestedData = $this->getExportNestedData($history);
            $data[]     = $nestedData;
        }

        return $data;
    }

    /**
     * Generates an array with export data fields
     *
     * @param $history
     *
     * @return array
     */
    public function getExportNestedData($history)
    {
        return [
            'Name'        => $history->user?->userName,
            'Email'       => $history->user?->email,
            'Title'       => $history->user?->business_title,
            'Location'    => $history->user?->city,
            'System Role' => $history->user?->userType?->name,
            'Status'      => $history->user?->status === 1 ? 'Active' : 'Inactive',
            'Date'        => $history->created_at,
        ];
    }
}
