<?php

namespace App\Services\Settings\History;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Settings\History\AssetsHistoryRepository;
use App\Services\Reports\AbstractReportService;

class SoftwareAssetHistoryService extends AbstractReportService
{
    /**
     * Constructor for the class.
     *
     * @param AssetsHistoryRepository $repository The asset history repository
     * @param CommonFilterRepository $commonFilterRepository The common filter repository
     * @param ReportOutputData $reportOutputData The report output data
     */
    public function __construct(
        protected AssetsHistoryRepository $repository,
        protected CommonFilterRepository $commonFilterRepository,
        protected ReportOutputData $reportOutputData,
    ) {}

    /**
     * Retrieves and processes data, then returns a compact array containing history and count.
     *
     * @return array
     */
    public function data()
    {
        $inputData = $this->getInputData();
        $assets    = $this->repository->getSoftwareAssetHistoryData();
        $assets    = $this->filterWithInputData($assets, $inputData);
        $count     = $assets->count();
        $history   = $this->reportOutputData->getOutputData($assets, ['id' => 'desc']);

        return compact('history', 'count');
    }

    /**
     * filterWithInputData function filters the query with the given input data.
     *
     * @param mixed $query The query to be filtered
     * @param array $inputData The input data used for filtering
     * @return mixed The filtered query
     */
    public function filterWithInputData($query, array $inputData)
    {
        $query = $this->commonFilterRepository->filterWithDirectDateRange($query, 'created_at', $inputData['from_date'], $inputData['to_date']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'action', $inputData['action']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'software_license_id', $inputData['software_asset']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'license_user_id', $inputData['license_user_id']);

        return $query;
    }

    /**
     * Get input data from the request array.
     *
     * @return array The input data array.
     */
    public function getInputData()
    {
        $requestData = request()->form ?? request()->all();

        return [
            'from_date'       => $requestData['from_date'] ?? '',
            'to_date'         => $requestData['to_date'] ?? '',
            'action'          => $requestData['action'] ?? '',
            'software_asset'  => $requestData['software_asset'] ?? '',
            'license_user_id' => $requestData['license_user_id'] ?? '',
        ];
    }

    /**
     * Retrieves nested data for the report.
     *
     * @param datatype $history description
     *
     * @return array
     */
    public function getReportNestedData($history)
    {
        $nestedData = [];
        $nestedData['report']['action']             = $nestedData['export']['Action'] = $history->action;
        $nestedData['report']['created_at']         = $nestedData['export']['Created At'] = $history->created_date;
        $nestedData['report']['summary']            = $nestedData['export']['Summary'] = $history->description;
        $nestedData['report']['modified_by']        = $nestedData['export']['Modified By'] = $history->user?->user_name;

        return $nestedData;
    }

    /**
     * Retrieves nested data for a given item and index.
     *
     * @param mixed $item The item to retrieve nested data for.
     * @param mixed $index The index of the item.
     *
     * @return array The nested data for the item.
     */
    public function getNestedData($item, $index)
    {
        $historyAction               = config('software-asset-history-action.' . $item->action);
        $commonData                  = $this->getReportNestedData($item);
        $nestedData                  = $commonData['report'];
        $nestedData['id']            = $index;
        $nestedData['action']        = "<span class='badge badge-" . ($historyAction['color'] ?? 'primary') . "'>" . ($historyAction['label'] ?? $item->action) . "</span>";
        $nestedData['modified_by']   = generateUserLink($item->user?->id, $item->user?->user_name);

        return $nestedData;
    }

    /**
     * Export data from the repository with input data filtering and ordering by ID.
     *
     * @return mixed
     */
    public function exportData()
    {
        $inputData = $this->getInputData();
        $assets = $this->repository->getSoftwareAssetHistoryData();
        $assets = $this->filterWithInputData($assets, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }
}
