<?php

namespace App\Services\Settings\History;

abstract class AbstractHistoryService
{
    abstract public function data();

    /**
     * Get input data from the request array.
     *
     * @return array The input data array.
     */
    public function getInputData()
    {
        $requestData = request()->form ?? request()->all();

        return [
            'from_date'   => $requestData['from_date'] ?? '',
            'to_date'     => $requestData['to_date'] ?? '',
            'location'    => $requestData['location_id'] ?? '',
            'user'        => $requestData['user'] ?? '',
            'action'      => $requestData['action'] ?? '',
            'asset'       => $requestData['asset'] ?? '',
            'modified_by' => $requestData['modified_by'] ?? '',
        ];
    }

    /**
     * Generating data table data from query result
     *
     * @param $query Query Result
     * @param $start Starting index
     * @param $data Collection variable that will return
     *
     * @return array
     */
    public function getReportData($query, $start, $data = [])
    {
        $parentIndex = $start;

        foreach ($query as $singleRecord) {
            $parentIndex++;
            $nestedData = $this->getNestedData($singleRecord, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieves nested data for the report.
     *
     * @param datatype $history
     *
     * @return array
     */
    public function getReportNestedData($history)
    {
        $nestedData                       = [];
        $nestedData['action']             = $history->action;
        $nestedData['created_at']         = $history->created_at;
        $nestedData['ticket_no']          = $history->ticket_no;
        $nestedData['summary']            = $history->description;
        $nestedData['user']               = ($history->newUser) ? $history->newUser?->user_name : $history->oldUser?->user_name;
        $nestedData['modified_by']        = ($history->user) ? $history->user?->user_name : $history->created_by;

        return $nestedData;
    }

    /**
     * Retrieves nested data for a given item and index.
     *
     * @param mixed $item The item to retrieve nested data for.
     * @param mixed $index The index of the item.
     *
     * @return array The nested data for the item.
     */
    public function getNestedData($item, $index)
    {
        $user                      = $item->newUser ?: $item->oldUser;
        $nestedData                = [];
        $historyAction             = config('history-action.' . $item->action);
        $nestedData                = $this->getReportNestedData($item);
        $nestedData['id']          = $index;
        $nestedData['action']      = "<span class='badge badge-" . ($historyAction['color'] ?? 'primary') . "'>" . ($historyAction['label'] ?? $item->action) . "</span>";
        $nestedData['user']        = generateUserLink($user?->id, $user?->user_name);
        $nestedData['modified_by'] = $item->user ? generateUserLink($item->user?->id, $item->user?->user_name) : $item->created_by;
        $nestedData['ticket_no']   = $this->ticketManagementService->getHistoryTicketUrl($item->ticket_service_provider, $item->ticket_no) ?: $item->ticket_no;

        return $nestedData;
    }
}
