<?php

namespace App\Services\Settings;

/**
 * Service Class for Asset type Export
 */
class AssetType
{

    /**
     * Retrieve asset type data, starting from a given index, and appending nested data for each asset type.
     *
     * @param array $assetTypes The list of asset types to process.
     * @param int   $start      The starting index.
     * @param array $data       The initial data array to which the nested data will be appended.
     *
     * @return array The updated data array with nested asset type data.
     */
    public function getAssetTypeData($assetTypes, $start, $data)
    {
        $parentIndex = $start;

        foreach ($assetTypes as $assetType) {
            $parentIndex++;
            $nestedData = $this->getNestedData($assetType);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieve nested data for a given asset type.
     *
     * @param object $assetTypes The asset type object containing the asset type details.
     *
     * @return array An associative array containing the asset type's name and the number of assets.
     */
    public function getNestedData($assetTypes)
    {
        $nestedData['name']         = $assetTypes->name ;
        $nestedData['no_of_assets'] = $assetTypes->assets_count;

        return $nestedData;
    }

    /**
     * Export the provided data as a CSV file and initiate a download.
     *
     * @param array $datas The data to be exported, structured as an array of arrays.
     *
     * @return \Illuminate\Http\Response The HTTP response containing the CSV file for download.
     */
    public function export($datas)
    {
        $headers = array(
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=file.csv",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        );

        $callback = function() use ($datas)
        {
            $file = fopen('php://output', 'w');
            fputcsv($file, array_keys($datas[0][0]));

            foreach($datas as $data) {
                foreach ($data as $data) {
                    fputcsv($file, $data);
                }

            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
