<?php

namespace App\Services\Security\Crowdstrike\Reports;

use App\Repositories\Security\Crowdstrike\CrowdstrikeReportRepository;
use Illuminate\Support\Str;

class SensorVulnerabilityService extends AbstractCrowdstrikeService
{


    protected $repository;

    /**
     * Constructs a new instance of the class.
     *
     * @param CrowdstrikeReportRepository $repository the repository to be used
     */
    public function __construct(CrowdstrikeReportRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Retrieves the sensor vulnerability data based on the given sort column and direction.
     *
     * @param string $sortColumn The column to sort the data by.
     * @param string $direction The direction in which to sort the data.
     * 
     * @return array Returns an array containing the count and the sensor vulnerability data.
     */
    public function data()
    {
        $sortData = $this->getSortOptions();
        $assetVulnerabilities = $this->repository->sensorVulnerabilityData();
        $requestData = request()->form;
        $assetVulnerabilities = $this->filterResults($assetVulnerabilities, $requestData);
        $count = $assetVulnerabilities->count();
        $assetVulnerabilities = $this->getOutputData($assetVulnerabilities, $sortData['column'], $sortData['direction']);

        return compact('count', 'assetVulnerabilities');
    }


    /**
     * Format the record for export as csv
     * @param object single record object
     * 
     * @return array formatted
     */
    public function getExportNestedData($assetVulnerability)
    {
        return [
            'Asset Tag #' => disableCSVInjection(optional($assetVulnerability->asset)->asset_tag ?? ''),
            'Serial #' => disableCSVInjection(optional($assetVulnerability->asset)->serial_no ?? ''),
            'Asset Status' => disableCSVInjection(optional($assetVulnerability->asset->assetStatus)->name ?? ''),
            'Hardware Standard' => disableCSVInjection(optional($assetVulnerability->asset->makeAndModel)->name ?? ''),
            'Tech Specs' => disableCSVInjection(optional($assetVulnerability->asset->technicalSpec)->details ?? ''),
            'User/Location' => disableCSVInjection($assetVulnerability->asset->user ? (optional($assetVulnerability->asset->user)->email) : ($assetVulnerability->asset->location ? (optional($assetVulnerability->asset->location)->room_name) : '')),
            'User Work Location' => disableCSVInjection(optional($assetVulnerability->asset->user)->city ?? ''),
            'CVE ID' => disableCSVInjection($assetVulnerability->cve_id),
            'ExPRT Rating' => disableCSVInjection($assetVulnerability->exprt_rating),
            'Severity' => disableCSVInjection($assetVulnerability->severity),
            'Product Name' => disableCSVInjection($assetVulnerability->product_name_normalized),
            'Product Version' => disableCSVInjection($assetVulnerability->product_name_version),
            'Days Open' => disableCSVInjection(get_day_diff_from_today($assetVulnerability->opened_at))
        ];
    }

    /**
     * Retrieves nested data for the given asset vulnerability.
     *
     * @param mixed $assetVulnerability The asset vulnerability object.
     * @param mixed $index The index value.
     * @param bool $export Indicates if the data should be exported.
     * @param bool $isAssetPageOrNot Indicates if request from a asset page.
     * 
     * @return array The nested data.
     */
    public function getNestedData($assetVulnerability, $index)
    {
        $nestedData = [];
        $commonData = $this->getExportNestedData($assetVulnerability);
        $assetLink  = route('assets.show', optional($assetVulnerability->asset)->id);
        $userLocationLink = $assetVulnerability->asset->user ? route('users.show', optional($assetVulnerability->asset->user)->id) : ($assetVulnerability->asset->location ? route('location.show', optional($assetVulnerability->asset->location)->id) : '#');
        $cveData = $this->repository->activeCveData()->where('cve_id', $assetVulnerability->cve_id)->first();
        $cveDetailLink  = route('active-cve.details', $cveData->id);

        $nestedData['id']   = $index;

        foreach ($commonData as $key => $value) {
            $nestedData[Str::slug($key, '_')] = $value;
        }

        $nestedData['asset_tag'] = "<a href=$assetLink>" . $nestedData['asset_tag'] . "</a>";
        $nestedData['serial'] = "<a href=$assetLink>" . $nestedData['serial'] . "</a>";
        $nestedData['userlocation'] = "<a href=$userLocationLink>" . $nestedData['userlocation'] . "</a>";
        $nestedData['cve_id'] = "<a href=$cveDetailLink>" . $nestedData['cve_id'] . "</a>";

        return $nestedData;
    }

    /**
     * Retrieves and exports data based on specified sorting criteria.
     *
     * @param string $sortColumn The column to sort the data by.
     * @param string $direction The direction of the sorting (ASC or DESC).
     * 
     * @return mixed The exported data.
     */
    public function exportData()
    {
        $sortData = $this->getSortOptions();
        $assetVulnerabilities = $this->repository->sensorVulnerabilityData();
        $requestData = request()->all();
        $assetVulnerabilities = $this->filterResults($assetVulnerabilities, $requestData);
        $assetVulnerabilities = $assetVulnerabilities->orderBy($sortData['column'], $sortData['direction']);

        return $assetVulnerabilities;
    }
}
