<?php

namespace App\Services\Security\Crowdstrike\Reports;

use App\Repositories\Security\Crowdstrike\CrowdstrikeReportRepository;
use Illuminate\Support\Str;

class ActiveCveService extends AbstractCrowdstrikeService
{

    protected $repository;
    
    /**
     * Constructs a new instance of the class.
     *
     * @param CrowdstrikeReportRepository $repository The repository to be injected.
     */
    public function __construct(CrowdstrikeReportRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Retrieves the active CVE data based on the specified filters.
     *
     * @return array Returns an array containing the count of active CVE data and the actual active CVE data.
     */
    public function data()
    {
        $activeCveData = $this->repository->activeCveData();
        $requestData = request()->form;
        $activeCveData = $this->filterResults($activeCveData, $requestData);
        // $count = $activeCveData->count();
        $count = count($activeCveData->get());
        $activeCveData = $this->getOutputData($activeCveData, 'id', 'desc');

        return compact('count', 'activeCveData');
    }

    /**
     * Format the record for export as csv
     * @param object single record object
     * 
     * @return array formatted
     */
    public function getExportNestedData($activeCveData)
    {
        return [
            'CVE ID' => disableCSVInjection($activeCveData->cve_id),
            'Base Score' => disableCSVInjection($activeCveData->base_score),
            'Severity' => disableCSVInjection($activeCveData->severity),
            'ExPRT Rating' => disableCSVInjection($activeCveData->exprt_rating),
            'Vulnerability Count' => disableCSVInjection($activeCveData->vulnerability_count),
            'CVE Published' => disableCSVInjection(parse_date_from_db_datetime($activeCveData->cve_published))
        ];
    }

    /**
     * Retrieves the nested data for a given active CVE.
     *
     * @param mixed $activeCveData The data of the active CVE.
     * @param mixed $index        The index of the nested data.
     * @param bool  $export       Determines if the data should be exported.
     *
     * @return array The nested data.
     */
    public function getNestedData($activeCveData, $index)
    {
        $nestedData = [];
        $commonData = $this->getExportNestedData($activeCveData);
        $cveLink  = route('active-cve.details', $activeCveData->id);
        $nestedData['id']   = $index;

        foreach ($commonData as $key => $value) {
            $nestedData[Str::slug($key, '_')] = $value;
        }
        $nestedData['cve_id'] = "<a href=$cveLink>" . $nestedData['cve_id'] . "</a>";
        
        return $nestedData;
    }

    /**
     * Export the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $activeCveData = $this->repository->activeCveData();
        $requestData = request()->all();
        $activeCveData = $this->filterResults($activeCveData, $requestData);
        $activeCveData = $activeCveData->orderBy('id', 'DESC');

        return $activeCveData;
    }
}