<?php

namespace App\Services\Revivn;

use Illuminate\Support\Str;
use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Revivn\RevivnPickupRepository;

class RevivnPickupReportService
{

    public function __construct(
        protected RevivnPickupRepository $repository,
        protected CommonFilterRepository $commonFilterRepository,
        protected ReportOutputData $reportOutputData
    ) {}

    /**
     * A description of the entire PHP function.
     *
     * @param datatype $paramname description
     * @throws Some_Exception_Class description of exception
     * @return Some_Return_Value
     */
    public function data($isDataTable = 0)
    {
        $inputData = $this->getInputData();
        $pickups = $this->repository->getData();
        $pickups = $this->filter($pickups, $inputData);

        if ($isDataTable == 1) {
            $count = $pickups->count();
            $pickups = $this->reportOutputData->getOutputData($pickups);

            return compact('pickups', 'count');
        }

        return $pickups->orderBy('created_at', 'desc');
    }

    /**
     * Get input data for filtering.
     *
     * @return array
     */
    public function getInputData()
    {
        $requestData = request()->form ?? request()->all();
        // Get input data for filtering.
        return [
            'pickupId' => isset($requestData['pickup_id']) ? $requestData['pickup_id'] : '',
            'state' => isset($requestData['state']) ? $requestData['state'] : '',
            'orderNumber' => isset($requestData['order_number']) ? $requestData['order_number'] : '',
            'scheduledDateFrom' => isset($requestData['scheduled_date_from']) ? $requestData['scheduled_date_from'] : '',
            'scheduledDateTo' => isset($requestData['scheduled_date_to']) ? $requestData['scheduled_date_to'] : '',

        ];
    }

    /**
     * Filters the pickups based on the given input data.
     *
     * @param array $pickups The pickups to filter.
     * @param mixed $inputData The input data to filter by.
     * @return array The filtered pickups.
     */
    public function filter($pickups, $inputData)
    {
        $pickups  = $this->commonFilterRepository->filterWithDirectFields($pickups, 'pickup_id', $inputData['pickupId']);
        $pickups  = $this->commonFilterRepository->filterWithDirectFields($pickups, 'state', $inputData['state']);
        $pickups  = $this->commonFilterRepository->filterWithDirectFields($pickups, 'order_number', $inputData['orderNumber']);
        $pickups = $this->commonFilterRepository->filterWithDirectDateRange($pickups, 'scheduled_date', $inputData['scheduledDateFrom'], $inputData['scheduledDateTo']);

        return $pickups;
    }

    /**
     * Get report data.
     *
     * @param datatype $pickups description
     * @param $start description
     * @param datatype $data description
     * @param datatype $isDataTable description
     * @return Some_Return_Value
     */
    public function getReportData($pickups, $start, $data, $isDataTable = 0)
    {
        $nestedData = [];
        $parentIndex = $start;

        foreach ($pickups as $pickup) {
            $parentIndex++;
            $nestedData = $this->getNestedData($pickup, $parentIndex, $isDataTable);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieves nested data.
     *
     * @param mixed $pickup The pickup to retrieve nested data for.
     * @param datatype $index description of the index parameter
     * @param datatype $isDataTable description of the isDataTable parameter
     * @return datatype
     */
    public function getNestedData($pickup, $index = 0, $isDataTable = 0)
    {
        $nestedData = [
            'Pickup ID' => disableCSVInjection($pickup->pickup_id),
            'State' => disableCSVInjection($pickup->state),
            'Order #' => disableCSVInjection($pickup->order_number),
            'Comments' => disableCSVInjection($pickup->comments),
            'Confirmed Time' => disableCSVInjection($pickup->confirmed_time),
            'Scheduled Date' => disableCSVInjection($pickup->scheduled_date),
        ];
        if ($isDataTable == 1) {
            $nestedData['id']   = $index;
            // Convert keys to slugs
            $nestedData = array_combine(
                array_map(fn($key) => Str::slug($key, '_'), array_keys($nestedData)),
                array_values($nestedData)
            );
            $pickUpUrl = '/inventory-control/revivn-assets?pickup=' . $nestedData['pickup_id'];
            $nestedData['pickup_id'] = "<a href=$pickUpUrl>" . $nestedData['pickup_id'] . "</a>";
        }

        return $nestedData;
    }
}
