<?php

namespace App\Services\Revivn;

use Illuminate\Support\Str;
use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Revivn\RevivnAssetsRepository;

class RevivnDevicesAutoCreatedService
{
    public function __construct(
        protected RevivnAssetsRepository $repository,
        protected ReportOutputData $reportOutputData,
        protected CommonFilterRepository $commonFilterRepository
    ) {
    }

    /**
     * Retrieves data and prepares it for output.
     *
     * @return array
     */
    public function data($isDataTable = 0)
    {
        $requestData = request()->form ?? request()->all();
        $inputData = $this->getInputData();
        $assets = $this->repository->getAutoCreatedDevices();
        $assets = $this->filter($assets, $inputData);

        if ($isDataTable == 1) {
            $count = $assets->count();
            $assets = $this->reportOutputData->getOutputData($assets);

            return compact('assets', 'count');
        }

        return $assets->orderBy('created_at', 'desc');
    }

    /**
     * Get input data for filtering.
     *
     * @return array
     */
    public function getInputData()
    {
        $requestData = request()->form ?? request()->all();
        // Get input data for filtering.
        return [
            'serial' => isset($requestData['serial_no']) ? $requestData['serial_no'] : '',
            'hardwareStandard' => isset($requestData['make_and_model']) ? $requestData['make_and_model'] : '',
            'manufacturer' => isset($requestData['manufacturer']) ? $requestData['manufacturer'] : '',
            'assetType' => isset($requestData['asset_type']) ? $requestData['asset_type'] : '',
            'technicalSpec' => isset($requestData['technical_spec']) ? $requestData['technical_spec'] : '',
            'pickupId' => isset($requestData['pickup_id']) ? $requestData['pickup_id'] : '',
            'hard_drive' => isset($requestData['hard_drive']) ? $requestData['hard_drive'] : '',
            'ram' => isset($requestData['ram']) ? $requestData['ram'] : '',
            'year' => isset($requestData['year']) ? $requestData['year'] : '',
            'date_added_from' => isset($requestData['date_added_from']) ? $requestData['date_added_from'] : '',
            'date_added_to' => isset($requestData['date_added_to']) ? $requestData['date_added_to'] : '',
        ];
    }

    /**
     * Making the query for Consumption report input filters
     */
    public function filter($assets, $inputData)
    {
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'serial_no', $inputData['serial']);
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $inputData['assetType']);
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'make_and_model_id', $inputData['hardwareStandard']);
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'technical_spec_id', $inputData['technicalSpec']);

        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'makeAndModel', 'manufacturer_id', $inputData['manufacturer']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'revivnAsset.revivnPickup', 'pickup_id', $inputData['pickupId']);

        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'revivnAsset', 'hard_drive', $inputData['hard_drive']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'revivnAsset', 'ram', $inputData['ram']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'revivnAsset', 'year', $inputData['year']);

        $assets = $this->commonFilterRepository->filterWithDirectDateRange($assets, 'created_at', $inputData['date_added_from'], $inputData['date_added_to']);

        return $assets;
    }


    /**
     * Generating data table/export data from qurey result
     * @param $assets Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     * @param $isDataTable 1 if data for data table, 0 for export
     *
     * @return array
     */
    public function getReportData($assets, $start, $data, $isDataTable = 0)
    {
        $nestedData = [];
        $parentIndex = $start;
        $data = array();
        foreach ($assets as $assets) {
            $parentIndex++;
            $nestedData = $this->getNestedData($assets, $parentIndex, $isDataTable);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieves nested data.
     *
     * @param mixed $asset The asset to retrieve nested data for.
     * @param $index Starting index
     * @param $isDataTable 1 if data for data table, 0 for export
     * @return array
     */
    public function getNestedData($asset, $index = 0, $isDataTable = 0)
    {

        $nestedData = [
            'Serial #' => disableCSVInjection($asset->serial_no),
            'Asset Type' => disableCSVInjection(optional($asset->assetType)->name),
            'Manufacturer' => disableCSVInjection(optional(optional($asset->makeAndModel)->manufacturer)->name),
            'Hardware Standard' => disableCSVInjection(optional($asset->makeAndModel)->name),
            'Tech Specs' => disableCSVInjection(optional($asset->technicalSpec)->details),
            'Hard Drive' => disableCSVInjection(optional($asset->revivnAsset)->hard_drive),
            'RAM' => disableCSVInjection(optional($asset->revivnAsset)->ram),
            'End Of Life Condition' => disableCSVInjection(optional($asset->revivnAsset)->end_of_life_condition),
            'Status' => disableCSVInjection(optional($asset->revivnAsset)->status),
            'Pickup ID' => disableCSVInjection(optional(optional($asset->revivnAsset)->revivnPickup)->pickup_id),
            'State' => disableCSVInjection(optional(optional($asset->revivnAsset)->revivnPickup)->state),
            'Order #' => disableCSVInjection(optional(optional($asset->revivnAsset)->revivnPickup)->order_number),
            'Year' => disableCSVInjection(optional($asset->revivnAsset)->year),
            'Date Added' => disableCSVInjection(optional($asset)->created_at),
        ];

        if ($isDataTable == 1) {
            $nestedData['id']   = $index;
            $assetLink = $asset ? route('assets.show', $asset->id) : '#';
            // Convert keys to slugs
            $nestedData = array_combine(
                array_map(fn ($key) => Str::slug($key, '_'), array_keys($nestedData)),
                array_values($nestedData)
            );
            $nestedData['serial'] = "<a href=$assetLink>" . $nestedData['serial'] . "</a>";
            $nestedData['pickup_id'] = '<span data-toggle="tooltip" data-placement="right" title="' . htmlspecialchars($nestedData['pickup_id']) . '">' . htmlspecialchars(substr($nestedData['pickup_id'], 0, 14)) . '...</span>';
            $nestedData['action_button'] = $this->getActionButton($asset);
        }

        return $nestedData;
    }

    /**
     * Set up Action Button
     * @param mixed $asset
     */
    public function getActionButton($asset)
    {
        if (auth()->user()->can('IT Assets Edit') && $asset) {
            return  "<a class='btn btn-link' href=''  data-toggle='modal' data-asset-id='" . $asset->id . "' data-target='#editModal' data-tooltip='tooltip' data-placement='left' title='Edit asset details'><i class='icon icon-n-edit'></i></a>";
        }
        return '';
    }
}
