<?php

namespace App\Services\Revivn;

use Illuminate\Support\Str;
use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Revivn\RevivnAssetsRepository;

class RevivnAssetsReportService
{

    public function __construct(
        protected RevivnAssetsRepository $repository,
        protected ReportOutputData $reportOutputData,
        protected CommonFilterRepository $commonFilterRepository
    ) {
    }

    /**
     * Retrieves data and prepares it for output.
     *
     * @return array
     */
    public function data($isDataTable = 0)
    {
        $inputData = $this->getInputData();
        $assets = $this->repository->getData();
        $assets = $this->filter($assets, $inputData);

        if ($isDataTable == 1) {
            $count = $assets->count();
            $assets = $this->reportOutputData->getOutputData($assets);

            return compact('assets', 'count');
        }

        return $assets->orderBy('created_at', 'desc');
    }

    /**
     * Get input data for filtering.
     *
     * @return array
     */
    public function getInputData()
    {
        $requestData = request()->form ?? request()->all();
        // Get input data for filtering.
        return [
            'serial' => isset($requestData['serial_no']) ? $requestData['serial_no'] : '',
            'hardwareStandard' => isset($requestData['make_and_model']) ? $requestData['make_and_model'] : '',
            'manufacturer' => isset($requestData['manufacturer']) ? $requestData['manufacturer'] : '',
            'assetType' => isset($requestData['asset_type']) ? $requestData['asset_type'] : '',
            'technicalSpec' => isset($requestData['technical_spec']) ? $requestData['technical_spec'] : '',
            'pickupId' => isset($requestData['pickup_id']) ? $requestData['pickup_id'] : '',
        ];
    }

    /**
     * Filters the given assets based on the input data.
     *
     * @param mixed $assets The assets to be filtered.
     * @param array $inputData The input data to filter the assets.
     * @return mixed The filtered assets.
     */
    public function filter($assets, $inputData)
    {
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'serial_no', $inputData['serial']);
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $inputData['assetType']);
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'make_and_model_id', $inputData['hardwareStandard']);
        $assets  = $this->commonFilterRepository->filterWithDirectFields($assets, 'technical_spec_id', $inputData['technicalSpec']);

        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'makeAndModel', 'manufacturer_id', $inputData['manufacturer']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'revivnAsset.revivnPickup', 'pickup_id', $inputData['pickupId']);

        return $assets;
    }

    /**
     * Generating data table/export data from qurey result
     * @param $assets Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     * @param $isDataTable 1 if data for data table, 0 for export
     *
     * @return array
     */
    public function getReportData($assets, $start, $data, $isDataTable = 0)
    {
        $nestedData = [];
        $parentIndex = $start;
        $data = array();
        foreach ($assets as $assets) {
            $parentIndex++;
            $nestedData = $this->getNestedData($assets, $parentIndex, $isDataTable);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieves nested data.
     *
     * @param mixed $asset The asset to retrieve nested data for.
     * @param $index Starting index
     * @param $isDataTable 1 if data for data table, 0 for export
     * @return array
     */
    public function getNestedData($asset, $index = 0, $isDataTable = 0)
    {

        $nestedData = [
            'Serial #' => disableCSVInjection($asset->serial_no),
            'Asset Type' => disableCSVInjection(optional($asset->assetType)->name),
            'Manufacturer' => disableCSVInjection(optional(optional($asset->makeAndModel)->manufacturer)->name),
            'Hardware Standard' => disableCSVInjection(optional($asset->makeAndModel)->name),
            'Tech Specs' => disableCSVInjection(optional($asset->technicalSpec)->details),
            'Condition' => disableCSVInjection(optional($asset->revivnAsset)->condition),
            'End Of Life Condition' => disableCSVInjection(optional($asset->revivnAsset)->end_of_life_condition),
            'Status' => disableCSVInjection(optional($asset->revivnAsset)->status),
            'Pickup ID' => disableCSVInjection(optional(optional($asset->revivnAsset)->revivnPickup)->pickup_id),
            'State' => disableCSVInjection(optional(optional($asset->revivnAsset)->revivnPickup)->state),
            'Order #' => disableCSVInjection(optional(optional($asset->revivnAsset)->revivnPickup)->order_number),
            'Scheduled Date' => disableCSVInjection(optional(optional($asset->revivnAsset)->revivnPickup)->scheduled_date),
        ];

        if ($isDataTable == 1) {
            $nestedData['id']   = $index;
            $assetLink = $asset ? route('assets.show', $asset->id) : '#';
            // Convert keys to slugs
            $nestedData = array_combine(
                array_map(fn ($key) => Str::slug($key, '_'), array_keys($nestedData)),
                array_values($nestedData)
            );
            $nestedData['serial'] = "<a href=$assetLink>" . $nestedData['serial'] . "</a>";
            $nestedData['pickup_id'] = '<span data-toggle="tooltip" data-placement="right" title="' . htmlspecialchars($nestedData['pickup_id']) . '">' . htmlspecialchars(substr($nestedData['pickup_id'], 0, 14)) . '...</span>';
        }

        return $nestedData;
    }
}
