<?php 

namespace App\Services;

use Cache;
use Illuminate\Support\InteractsWithTime;

class ResetPassword
{
	use InteractsWithTime;
	
	public function __construct(Cache $cache)
    {
        $this->cache = $cache;
    }
	/**
     * Determine if the given key has been "accessed" too many times.
     *
     * @param  string  $key
     * @param  int  $maxAttempts
     * @return bool
     */
    public function tooManyAttempts($key, $maxAttempts)
    {
        if ($this->attempts($key) >= $maxAttempts) {
            if (Cache::has($key.':timer')) {
                return true;
            }

            $this->resetAttempts($key);
        }

        return false;
    }
    /**
     * Increment the counter for a given key for a given decay time.
     *
     * @param  string  $key
     * @param  float|int  $decayMinutes
     * @return int
     */
    public function hit($key, $decaySeconds = 60)
    {
    	Cache::add($key.':timer', $this->availableAt($decaySeconds), $decaySeconds);
        
        $added = Cache::add($key, 0, $decaySeconds);

        $hits = (int) Cache::increment($key);

        if (! $added && $hits == 1) {
            Cache::put($key, 1, $decaySeconds);
        }

        return $hits;
    }
    /**
     * Get the number of attempts for the given key.
     *
     * @param  string  $key
     * @return mixed
     */
    public function attempts($key)
    {
        return Cache::get($key, 0);
    }
    /**
     * Reset the number of attempts for the given key.
     *
     * @param  string  $key
     * @return mixed
     */
    public function resetAttempts($key)
    {
        return Cache::forget($key);
    }
    /**
     * Get the number of retries left for the given key.
     *
     * @param  string  $key
     * @param  int  $maxAttempts
     * @return int
     */
    public function retriesLeft($key, $maxAttempts)
    {
        $attempts = $this->attempts($key);

        return $maxAttempts - $attempts;
    }

    /**
     * Clear the hits and lockout timer for the given key.
     *
     * @param  string  $key
     * @return void
     */
    public function clear($key)
    {
        $this->resetAttempts($key);

        Cache::forget($key.':timer');
    }

    /**
     * Get the number of seconds until the "key" is accessible again.
     *
     * @param  string  $key
     * @return int
     */
    public function availableIn($key)
    {
        return Cache::get($key.':timer') - $this->currentTime();
    }
}