<?php

namespace App\Services\Reports;

use App\Repositories\Reports\WarrantyExpirationDateRepository;
use App\Repositories\CommonFilterRepository;
use App\Http\Responses\ReportOutputData;

/**
 * WarrantyExpirationDateService - service class for assets by warranty report
 */
class WarrantyExpirationDateService  extends AbstractReportService
{
    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * __construct
     *
     * @param  WarrantyExpirationDateRepository $repository
     * @param CommonFilterRepository $commonFilterRepository The commonFilterRepository instance.
     * @param ReportOutputData $reportOutputData The reportOutputData instance
     */
    public function __construct(WarrantyExpirationDateRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * filter function used in assets warrenty reports controller
     *
     * @return array
     */
    public function data()
    {
        $result     = $this->repository->getAssetsByWarrantyData();
        $inputData  = $this->getInputData();
        $result     = $this->filterResults($result, $inputData);
        $count      = $result->get()->count();

        $result     = $this->reportOutputData->getOutputData($result, ['assets.created_at' => 'asc']);

        return compact('result', 'count');
    }

    /**
     * Taking the input data
     * @param Http\Request
     * @return Array
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();

        return [
            'serial_no' => $requestedData['serial_no'] ?? '',
            'asset_tag' => $requestedData['asset_tag'] ?? '',
            'make_and_model_id' => $requestedData['make_and_model'] ?? '',
            'technical_spec_id'  => $requestedData['technical_spec'] ?? '',
            'asset_status_id'  => $requestedData['status'] ?? '',
            'assetUser'  => $requestedData['user'] ?? '',
            'warrantyStartFrom'  => $requestedData['warranty_start_date_from'] ?? '',
            'warrantyStartTo'  => $requestedData['warranty_start_date_to'] ?? '',
            'warrantyEndFrom'  => $requestedData['warranty_end_date_from'] ?? '',
            'warrantyEndTo'  => $requestedData['warranty_end_date_to'] ?? '',
            'daysToExpire'  => $requestedData['days_to_expire'] ?? '',
            'location'  => $requestedData['location'] ?? ''
        ];
    }

    /**
     * Filters the results based on the input data.
     *
     * @param array $assets The array of assets to filter.
     * @param array $inputData The input data used for filtering.
     * @return object The filtered array of assets.
     */
    public function filterResults($assets, $inputData)
    {
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', 'id', $inputData['assetUser']);
        $assets = $this->commonFilterRepository->filterWithDirectDateRange($assets, 'created_at', $inputData['warrantyStartFrom'], $inputData['warrantyStartTo']);
        $assets = $this->commonFilterRepository->filterWithDirectDateRange($assets, 'warranty_end_date', $inputData['warrantyEndFrom'], $inputData['warrantyEndTo']);
        $assets = $this->repository->filterWithExpiryDate($assets, $inputData['daysToExpire']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'serial_no', $inputData['serial_no']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_tag', $inputData['asset_tag']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'make_and_model_id', $inputData['make_and_model_id']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'technical_spec_id', $inputData['technical_spec_id']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_status_id', $inputData['asset_status_id']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', 'city', $inputData['assetUser']);

        return $assets;
    }

    /**
     * Returns an array of nested data for exporting.
     *
     * @param mixed $asset The asset object.
     * @return array The nested data array.
     */
    public function getReportNestedData($asset)
    {
        $nestedData['export']['Serial #']            = $nestedData['report']['serial_no'] = $asset->serial_no;
        $nestedData['export']['Asset Tag #']           = $nestedData['report']['asset_tag'] = $asset->asset_tag;
        $nestedData['export']['Hardware Standard']   = $nestedData['report']['hardware_standard'] = $asset->makeAndModel ? $asset->makeAndModel->makeModelName : '';
        $nestedData['export']['Technical Specs']     = $nestedData['report']['tech_specs'] = $asset->technicalSpec ? $asset->technicalSpec->details : '';
        $nestedData['export']['location']            = $nestedData['report']['location'] = $asset->user ? ($asset->user->first_name . ' ' . $asset->user->last_name) : ($asset->location ? $asset->location->room_name : '');
        $nestedData['export']['User Location']       = $nestedData['report']['user_location']  = $asset->user ? ($asset->user->city ? $asset->user->city : "") : "";
        $nestedData['export']['Asset Status']        = $nestedData['report']['asset_status']  = $asset->assetStatus ? $asset->assetStatus->name : '';
        $nestedData['export']['Warranty Start Date'] = $nestedData['report']['warranty_start_date'] = $asset->created_at;
        $nestedData['export']['Warranty End Date']   = $nestedData['report']['warranty_end_date'] = $asset->warranty_end_date;
        $nestedData['export']['Days to expire']      = $nestedData['report']['days_to_expire'] = !past_date_or_not($asset->warranty_end_date) ? calculate_diff_in_days_from_today($asset->warranty_end_date) : 0;

        return $nestedData;
    }

    /**
     * getNestedData - get data for display in report table
     *
     * @param  mixed $asset
     * @param  mixed $index
     * @return array
     */
    public function getNestedData($asset, $index)
    {
        $commonData = $this->getReportNestedData($asset);
        $nestedData = $commonData['report'];
        $assetLink  = route('assets.show', $asset->id);
        $nestedData['id']   = $index;
        $nestedData['asset_tag'] = "<a href=$assetLink>" . $nestedData['asset_tag'] . "</a>";
        $nestedData['serial_no'] = "<a href=$assetLink>" . $nestedData['serial_no'] . "</a>";

        return $nestedData;
    }

    /**
     * getReportData
     *
     * @return object
     */
    public function exportData()
    {
        $assets = $this->repository->getAssetsByWarrantyData();
        $inputData  = $this->getInputData();
        $assets = $this->filterResults($assets, $inputData);
        $assets = $assets->orderBy('assets.created_at');

        return $assets;
    }
}
