<?php

namespace App\Services\Reports;

use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Repositories\CommonFilterRepository;
use App\Http\Responses\ReportOutputData;
use App\Repositories\Reports\UsersToAssetsRatioRepository;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use App\Services\ReportGraphAndDataExport;

class UsersToAssetsRatioService  extends AbstractReportService
{

    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Constructs a new instance of the class.
     *
     * @param UsersToAssetsRatioRepository $repository the asset type to user repository
     * @param CommonFilterRepository $commonFilterRepository the common filter repository
     * @param ReportOutputData $reportOutputData the report output data
     */
    public function __construct(UsersToAssetsRatioRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Retrieves the filter data options for the current request.
     *
     * @return array The filter data options.
     */
    public function getFilterDataOptions()
    {
        $filterDataOptions['assetTypes'] = AssetType::withoutSlug(['accessories']);
        $filterDataOptions['assetStatus'] = AssetStatus::WithUser()->get();
        $filterDataOptions['workLocations'] = request('work_location') ?? [];

        return $filterDataOptions;
    }

    /**
     * Generates the function comment for the formattedData function.
     *
     * @param mixed $request The request data.
     * @return array The formatted data.
     */
    public function formattedData($request)
    {
        $inputData = $this->getInputData();
        extract($inputData);

        if ($userStatus == 2) {
            $userStatus = 0;
        }
        $formattedData = [];
        $assets = $this->repository->getAssetTypeToUserFormattedData();
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $assetTypeId);

        if ($userStatus == '0' || $userStatus == '1') {
            $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', 'status', $userStatus);
        }
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_status_id', $assetStatusId);

        $userFilter = ['city' => $workLocation, 'department_id' => $departmentId, 'employee_type_id' => $userType];

        foreach ($userFilter as $dbField => $value) {
            $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', $dbField, $value);
        }

        $assets = $assets->whereNotNull('user_id')->where('user_id', '!=', 0)->groupBy('user_id')->get();
        $data = [];

        foreach ($assets as  $asset) {
            $data[] = $asset->cnt;
        }
        $data = array_count_values($data);

        $dataAboveFifteen = $this->getDataAboveFifteen($data);

        for ($i = 1; $i < 15; ++$i) {
            $formattedData[] = isset($data[$i]) ? $data[$i] : 0;
        }
        $formattedData[] = $dataAboveFifteen;

        if (!empty($assetFrom)) {
            $assetFrom = $assetFrom - 1;

            if (isset($formattedData)) {

                foreach ($formattedData as $key => $value) {

                    if ($key < $assetFrom) {
                        $formattedData[$key] = 0;
                    }
                }
            }
        }

        if (!empty($assetTo)) {
            $assetTo = $assetTo - 1;

            if (isset($formattedData)) {

                foreach ($formattedData as $key => $value) {

                    if ($key > $assetTo) {
                        $formattedData[$key] = 0;
                    }
                }
            }
        }
        return $formattedData;
    }

    /**
     * Get Input data
     * @return Array
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();

        return [
            'assetTypeId' => $requestedData['asset-type'] ?? null,
            'assetStatusId' => $requestedData['asset-status'] ?? null,
            'userStatus' => $requestedData['user-status'] ?? null,
            'assetFrom' =>  $requestedData['assetsCount'] ?? ($requestedData['asset-from'] ?? 1),
            'assetTo' => $requestedData['assetsCount'] ?? ($requestedData['asset-to'] ?? null),
            'userType' => $requestedData['employee_type'] ?? null,
            'regionId' => $requestedData['region'] ??null,
            'workLocation' => $requestedData['work_location'] ?? null,
            'departmentId' => $requestedData['department'] ?? null,
        ];
    }

    /**
     * Computes the sum of values in the given array that have a key greater than 14.
     *
     * @param array $data The input array.
     * @return int The sum of values with keys greater than 14.
     */
    public function getDataAboveFifteen($data)
    {
        $fifteenPlus = 0;

        foreach ($data as $key => $value) {
            if ($key > 14) {
                $fifteenPlus += $value;
            }
        }

        return $fifteenPlus;
    }



    /**
     * Making query for the report data.
     *
     * @return array
     */
    public function data()
    {
        $result = $this->repository->getAssetTypeToUserData();
        $inputData = $this->getInputData();
        $result = $this->filterWithInputData($result, $inputData);

        if ($inputData['assetFrom'] != null || $inputData['assetTo'] != null) {
            $assetsUsers = $this->getUsersWithMulipleAssets($inputData);
            $result = $result->whereIn('user_id', $assetsUsers);
        }

        $result = $result->whereNotNull('user_id')->where('user_id', '!=', 0);
        $count = $result->count();
        $result = $this->reportOutputData->getOutputData($result, ['users.email' => 'asc']);

        return compact('result', 'count');
        
    }

    /**
     * Making the query for input filters.
     *
     * @param object $stocks
     * @param array $inputData
     *
     * @return object
     */
    public function filterWithInputData($assets, $inputData)
    {
        extract($inputData);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $assetTypeId);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_status_id', $assetStatusId);
        $userStatus = ($userStatus == 2) ? 0 : $userStatus;

        if ($userStatus == '0' || $userStatus == '1') {
            $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', 'status', $userStatus);
        }
        $userFilter = ['city' => $workLocation, 'department_id' => $departmentId, 'employee_type_id' => $userType];

        foreach ($userFilter as $dbField => $value) {
            $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', $dbField, $value);
        }

        return $assets;
    }

    /**
     * Retrieves users with multiple assets based on the input data.
     *
     * @param array $inputData The input data containing asset filters.
     * @return array The array of user IDs.
     */
    public function getUsersWithMulipleAssets($inputData)
    {
        if ($inputData['assetFrom'] == '15+') {
            $inputData['assetFrom'] = 15;
        }

        if ($inputData['assetTo'] == '15+') {
            $inputData['assetTo'] = 15;
        }
        $assets = $this->repository->getAssetTypeToUserData();
        $assetsUsers = $this->filterWithInputData($assets, $inputData);
        $assetsUsers->groupBy('assets.user_id');

        if ($inputData['assetFrom'] != null && $inputData['assetFrom'] != 15) {
            $assetsUsers = $assetsUsers->havingRaw('COUNT(*) >=' . $inputData['assetFrom']);
        }

        if ($inputData['assetFrom'] == 15) {
            $assetsUsers = $assetsUsers->havingRaw('COUNT(*) >=' . $inputData['assetFrom']);
        }

        if ($inputData['assetTo'] != null && $inputData['assetTo'] != 15) {
            $assetsUsers = $assetsUsers->havingRaw('COUNT(*) <=' . $inputData['assetTo']);
        }
        $assetsUsers = $assetsUsers->pluck('user_id');

        return $assetsUsers;
    }

    /**
     * Making the query for export.
     *
     * @return object
     */
    public function exportData()
    {
        $assets = $this->repository->getAssetTypeToUserData();
            $inputData = $this->getInputData();
            $assets = $this->filterWithInputData($assets, $inputData);


            if ($inputData['assetFrom'] != null || $inputData['assetTo'] != null) {
                $assetsUsers = $this->getUsersWithMulipleAssets($inputData);
                $assets = $assets->whereIn('user_id', $assetsUsers);
            }

            $assets = $assets->whereNotNull('user_id')->where('user_id', '!=', 0);
            $assets = $assets->orderBy('users.email', 'asc');

            return $assets;
    }

    /**
     * Creating array for export data.
     *
     * @param $item Query Result
     *
     * @return array
     */
    public function getReportNestedData($asset)
    {
        $nestedData['report']['asset_tag'] = $nestedData['export']['Asset Tag #'] = disableCSVInjection($asset->asset_tag);
        $nestedData['report']['serial_number'] = $nestedData['export']['Serial #'] = disableCSVInjection($asset->serial_no);
        $nestedData['report']['type'] = $nestedData['export']['Asset Type'] = disableCSVInjection($asset->assetType ? $asset->assetType->name : '');
        $nestedData['report']['hardware_standard'] = $nestedData['export']['Hardware Standard'] = disableCSVInjection($asset->makeAndModel ? $asset->makeAndModel->makeModelName : '');
        $nestedData['report']['technical_spec'] = $nestedData['export']['Technical Specs'] = disableCSVInjection($asset->technicalSpec ? $asset->technicalSpec->details : '');
        $nestedData['report']['last_seen'] = $nestedData['export']['Last Seen'] = disableCSVInjection($asset->workspaceOne ? $asset->workspaceOne->last_seen : '');
        $nestedData['report']['asset_status'] = $nestedData['export']['Asset Status'] = disableCSVInjection($asset->assetStatus ? $asset->assetStatus->name : '');
        $nestedData['report']['user_location'] = $nestedData['export']['User/Location'] = disableCSVInjection($asset->user ? ($asset->user?->user_name) : ($asset->location ? $asset->location->room_name : ''));
        if ($asset->user && $asset->location) {
            $nestedData['report']['user_location'] = $nestedData['export']['User/Location'] = disableCSVInjection(($asset->user?->first_name . ' ' . $asset->user?->last_name) . ' / ' . $asset->location->room_name);
        }
        $nestedData['report']['email'] = $nestedData['export']['Email'] = disableCSVInjection($asset->user ? $asset->user->email : '');
        $nestedData['report']['imei'] = $nestedData['export']['IMEI'] = disableCSVInjection($asset->imei);
        $nestedData['report']['employee_id'] = $nestedData['export']['Employee ID'] = disableCSVInjection($asset->user ? $asset->user->employee_id : '');
        $nestedData['report']['business_title'] = $nestedData['export']['Title'] = disableCSVInjection($asset->user ? optional($asset->user->position)->name : '');
        $nestedData['report']['department'] = $nestedData['export']['Department'] = disableCSVInjection($asset->user ? optional($asset->user->department)->name : '');
        $nestedData['report']['work_location'] = $nestedData['export']['Work Location'] = disableCSVInjection($asset->user ? $asset->user->city : '');
        $nestedData['report']['legal_hold'] = $nestedData['export']['Legal Hold'] = $asset->user ? ($asset->user->legalhold == 1 ? 'Yes' : 'No') : '';
        $nestedData['report']['hire_date'] = $nestedData['export']['Hire Date'] = disableCSVInjection($asset->user ? $asset->user->hire_date : '');
        $nestedData['report']['terminated_date'] = $nestedData['export']['Terminated Date'] = disableCSVInjection($asset->user ? $asset->user->terminated_date : '');
        $nestedData['report']['status'] = $nestedData['export']['Status'] = $asset->user ? ($asset->user->status == 1 ? 'Active' : 'Inactive') : '';
        $nestedData['report']['date_deployed'] = $nestedData['export']['Date Deployed'] = disableCSVInjection($asset->date_deployed);
        $nestedData['report']['lease_start_date'] = $nestedData['export']['Lease Start Date'] = disableCSVInjection($asset->lease_start_date);
        $nestedData['report']['lease_end_date'] = $nestedData['export']['Lease End Date'] = disableCSVInjection($asset->lease_end_date);
        $nestedData['report']['loaner_return_date'] = $nestedData['export']['Expected Return Date'] = disableCSVInjection($asset->loaner_return_date);
        $nestedData['report']['loaner_retention_date'] = $nestedData['export']['Loaner Retention Date'] = disableCSVInjection($asset->loaner_retention_date);
        $nestedData['report']['ticket_no'] = $nestedData['export']['Ticket Number #'] = disableCSVInjection($asset->ticket_no);
        $nestedData['report']['carrier'] = $nestedData['export']['Mobile Carrier'] = disableCSVInjection($asset->carrier ? $asset->carrier->name : '');
        $nestedData['report']['parent_asset_tag'] = $nestedData['export']['Parent Asset'] = disableCSVInjection($asset->parentAsset ? $asset->parentAsset->asset_tag : '');
        $nestedData['report']['linked_date'] = $nestedData['export']['Linked Date'] = disableCSVInjection($asset->linked_date);
        $nestedData['report']['lost_date'] = $nestedData['export']['Lost Date'] = disableCSVInjection($asset->lost_date);
        $nestedData['report']['cancelled_date'] = $nestedData['export']['Cancelled Date'] = disableCSVInjection($asset->cancelled_date);
        $nestedData['report']['end_of_life_date'] = $nestedData['export']['End Of Life Date'] = disableCSVInjection($asset->end_of_life_date);
        // $nestedData['report']['wipe_confirmation'] = $nestedData['export']['Wipe Confirmation'] = disableCSVInjection($asset->wipe_confirmation);
        $nestedData['report']['donation_certificate'] = $asset->donation_certificate ? route('downloadcertificate', ['file' => str_replace('/', '-', $asset->donation_certificate)]) : '';
        $nestedData['report']['age'] = $nestedData['export']['Asset Age'] = $asset->AssetAge;
        $nestedData['report']['created_at'] = $nestedData['export']['Created At'] = disableCSVInjection($asset->created_at);
        $nestedData['report']['modified_date'] = $nestedData['export']['Last Modified Date'] = disableCSVInjection($asset->latestAssetHistory ? $asset->latestAssetHistory->updated_at : '');
        $nestedData['report']['modified_by'] = $nestedData['export']['Last Modified By'] = disableCSVInjection($asset->latestAssetHistory ? optional($asset->latestAssetHistory->user)->user_name : '');

        return $nestedData;
    }
    /*
    * Retrieves nested data for a given asset and index.
    *
    * @param mixed $asset The asset object.
    * @param int $index The index of the nested data.
    *
    * @return array The nested data.
    */
    public function getNestedData($asset, $index)
    {
        $nestedData = [];
        $commonData = $this->getReportNestedData($asset);
        $nestedData = $commonData['report'];
        $nestedData['id']   = $index;
        $nestedData['asset_tag'] = generateAssetLink($asset->id, $asset->asset_tag);
        $nestedData['serial_number'] = generateAssetLink($asset->id, $asset->serial_no);
        $nestedData['modified_by'] = generateUserLink($asset->latestAssetHistory?->user?->id, $asset->latestAssetHistory?->user?->user_name);
        $userAndLocation = generateUserLink($asset->user?->id, $asset->user?->user_name);

        if ($asset->user && $asset->location) {
            $userAndLocation .= ' / ' . generateLocationLink($asset->location->id, $asset->location->room_name);
        }
        $nestedData['user_location'] = $userAndLocation;

        return $nestedData;
    }
   
    /**
     * Retrieves the graph data based on the input parameters.
     *
     * @return array The formatted data for the graph.
     */
    public function getGraphData()
    {
        $inputData = $this->getInputData();
        extract($inputData);

        if ($userStatus == 2) {
            $userStatus = 0;
        }
        
        $formattedData = [];

        $assets = $this->repository->getAssetTypeToUserFormattedData();
        
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $assetTypeId);

        if ($userStatus == '0' || $userStatus == '1') {
            $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', 'status', $userStatus);
        }
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_status_id', $assetStatusId);

        $userFilter = ['city' => $workLocation, 'department_id' => $departmentId, 'employee_type_id' => $userType];

        foreach ($userFilter as $dbField => $value) {
            $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', $dbField, $value);
        }

        $assets = $assets->whereNotNull('user_id')->where('user_id', '!=', 0)->groupBy('user_id')->get();
        $data = [];

        foreach ($assets as  $asset) {
            $data[] = $asset->cnt;
        }
        $data = array_count_values($data);

        $dataAboveFifteen = $this->getDataAboveFifteen($data);

        for ($i = 1; $i < 15; ++$i) {
            $formattedData[] = isset($data[$i]) ? $data[$i] : 0;
        }
        $formattedData[] = $dataAboveFifteen;

        if (!empty($assetFrom)) {
            $assetFrom = $assetFrom - 1;

            if (isset($formattedData)) {

                foreach ($formattedData as $key => $value) {

                    if ($key < $assetFrom) {
                        $formattedData[$key] = 0;
                    }
                }
            }
        }

        if (!empty($assetTo)) {
            $assetTo = $assetTo - 1;

            if (isset($formattedData)) {

                foreach ($formattedData as $key => $value) {

                    if ($key > $assetTo) {
                        $formattedData[$key] = 0;
                    }
                }
            }
        }

        return $formattedData;
    }

    /**
	 * Export the graph and data list to Excel.
	 *
	 * @param array $dataCollection The collection of data.
	 */
	public function exportGraphAndDataListToExcel(array $dataCollection)
	{
		$name = str_replace(' ', '_', request('name'));
		$name = str_replace('/', '_', $name);
		$sheetTitle = str_replace(' ', '_', request('sheettitle'));
		$sheetTitle = str_replace('/', '_', $sheetTitle);

		$headings = array_keys($dataCollection[0][0]);
		$image = storage_path('app/' . request('image'));
		$filename = request("filename") . time() . '.xlsx';
		$excel = Excel::download(new ReportGraphAndDataExport($dataCollection, $image, $headings, $sheetTitle), $filename, \Maatwebsite\Excel\Excel::XLSX);
		$img = explode('public/', request('image'));
		Storage::disk('public')->delete($img[1]);

		return $excel;
	}
}
