<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Models\AssetStatus;
use App\Repositories\Reports\AssetsUsedRepository;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Reports\UseTaxReportRepository;
use Illuminate\Support\Facades\Cache;

/**
 * Service class for Use Tax report
 */
class UseTaxService extends AbstractReportService
{
    /**
     * Initializes a new instance of the class.
     *
     * @param UseTaxReportRepository $repository The use tax report repository.
     * @param AssetsUsedRepository $assetsUsedRepository The assets used repository.
     * @param AssetsAssignedService $assetAssignedService The asset assigned service.
     * @param ReportOutputData $reportOutputData The report output data.
     * @param CommonFilterRepository $commonFilterRepository The common filter repository.
     */
    public function __construct(
        protected UseTaxReportRepository $repository,
        protected AssetsUsedRepository $assetsUsedRepository,
        protected AssetsAssignedService $assetAssignedService,
        protected ReportOutputData $reportOutputData,
        protected CommonFilterRepository $commonFilterRepository
    ) {
        $this->repository = $repository;
        $this->assetAssignedService = $assetAssignedService;
        $this->assetsUsedRepository = $assetsUsedRepository;
        $this->commonFilterRepository = $commonFilterRepository;
    }

    /**
     * filter
     *
     * @return void
     */
    public function data()
    {
        $requestedData = request()->form;
        $inputData = $this->getInputData($requestedData);
        $result = $this->repository->getAssets();
        $result = $this->filterWithInputData($result, $inputData);
        $count  = $result->count();
        $result = $this->reportOutputData->getOutputData($result, ['id' => 'asc']);
        $reAssignedAssets = $this->assetAssignedService->getAssignedAssets('');
        $result =  $this->assetsUsedRepository->mapAssetsWithAssetCount($result, $reAssignedAssets['count']);

        return compact('result', 'count');
    }

    /**
     * Filters the given assets with the provided input data.
     *
     * @param array $assets The array of assets to be filtered.
     * @param array $inputData The input data used for filtering the assets.
     * @return The filtered assets.
     */
    public function filterWithInputData($assets, $inputData)
    {
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_tag', $inputData['asset_tag']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'serial_no', $inputData['serial_no']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_status_id', $inputData['asset_status_id']);
        $assets = $this->repository->filterWithAssetType($assets);
        $assets = $this->repository->filterWithAssetStatus($assets);

        return $assets;
    }

    /**
     * Retrieves and returns the input data from the given array.
     *
     * @param array $requestedData The array containing the requested data.
     * @return array The input data retrieved from the array.
     */
    public function getInputData(array $requestedData): array
    {
        return [
            'asset_tag'          => $requestedData['asset_tag'] ?? '',
            'serial_no'          => $requestedData['serial_no'] ?? '',
            'asset_status_id'    => $requestedData['status'] ?? '',
        ];
    }

    /**
     * Retrieves the nested data for generating a report.
     *
     * @param $asset The asset object for which the report is being generated.
     */
    public function getReportNestedData($asset)
    {
        $nestedData = [];
        $nestedData['report']['asset_tag']          = $nestedData['export']['Asset Tag #'] = $asset->asset_tag;
        $nestedData['report']['serial_no']          = $nestedData['export']['Serial #'] = $asset->serial_no;
        $nestedData['report']['type']               = $nestedData['export']['Asset Type'] = $asset->assetType ? $asset->assetType->name : '';
        $nestedData['report']['hardware_standard']  = $nestedData['export']['Hardware Standard'] = $asset->makeAndModel ? $asset->makeAndModel->makeModelName : '';
        $nestedData['report']['asset_age']          = $nestedData['export']['Asset Age'] = (validateDate($asset->created_at)) ? age($asset->created_at) : '';
        $nestedData['report']['purchase_date']      = $nestedData['export']['Purchase Date'] = $asset->created_at;
        $nestedData['report']['status']             = $nestedData['export']['Asset Status'] = $asset?->assetStatus?->name;
        $nestedData['report']['origin_location']    = $nestedData['export']['Origin Location'] = $this->getOriginLocationFromAssetHistory($asset);
        $nestedData['report']['location_user']      = $nestedData['export']['User/Location'] = $asset->user ? ($asset?->user?->user_name) : $asset->location?->room_name;
        $nestedData['report']['days_at_location']   = $nestedData['export']['Days At Current Location'] = $this->repository->daysAtCurrentLocation($asset->latestAssetHistory, $asset->created_at,);
        $nestedData['report']['work_location']      = $nestedData['export']['User Work Location'] = $asset->user && ($asset->user->status == 1) ? $asset->user->city : '';
        $nestedData['report']['department']         = $nestedData['export']['Department'] = $asset?->user?->department?->name;
        $nestedData['report']['hire_date']          = $nestedData['export']['Hire Date'] = $asset?->user?->hire_date;
        $nestedData['report']['reassigned_count']   = $nestedData['export']['#of Times Reassigned'] = $asset->count;

        return $nestedData;
    }

    /**
     * Create each row of output data
     * @params $assets,$index,$type
     * @return array
     * */
    public function getNestedData($item, $index)
    {
        $nestedData = [];
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id'] = $index;
        $nestedData['asset_tag'] = generateAssetLink($item->id, $item->asset_tag);
        $nestedData['serial_no'] = generateAssetLink($item->id, $item->serial_no);
        $nestedData['location_user'] = $item->user ? generateUserLink($item->user->id, $item->user?->user_name) : generateLocationLink($item->location->id, $item->location?->room_name);

        return $nestedData;
    }
    /**
     * getExportData
     *
     * @return mixed
     */
    public function exportData()
    {
        $requestedData = request()->all();
        $inputData = $this->getInputData($requestedData);
        $assets = $this->repository->getAssets();
        $assets = $this->filterWithInputData($assets, $inputData);
        $assets = $assets->orderBy('id');

        return $assets;
    }

    /**
     * get origin location of asset based on asset history
     *
     * @return string location name
     */
    public function getOriginLocationFromAssetHistory(mixed $asset)
    {
        $assignedStatus = Cache::remember('assigned-status', 86400, fn () => AssetStatus::where('slug', 'assigned')->first());

        if ($asset->asset_status_id == $assignedStatus->id) {
            $shiStatus = Cache::remember('shi-in-transit', 86400, fn () => AssetStatus::select('name')->whereSlug('shi_in_transit')->first());

            $assetHistory = $asset->latestAssetHistory ?? '';

            if (!$assetHistory || $assetHistory->old_value != $shiStatus?->name) {
                return optional($asset->originLocation)->room_name;
            }

            if ($assetHistory->new_value == $assignedStatus->name) {
                return config('shi-location.label');
            }
        }
        return optional($asset->originLocation)->room_name;
    }

    /**
     * Retrieves the asset status.
     *
     * @return mixed
     */
    public function getAssetStatus()
    {
        return $this->repository->getAssetStatus();
    }

    /**
     * Maps assets with their corresponding asset count.
     *
     * @param array $assets The array of assets.
     * @return mixed The mapped assets with their asset count.
     */
    public function mapAssetsWithAssetCount($assets)
    {
        $reAssignedAssets = $this->assetAssignedService->getAssignedAssets('');

        return $this->assetsUsedRepository->mapAssetsWithAssetCount($assets, $reAssignedAssets['count']);
    }
}
