<?php

namespace App\Services\Reports;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Repositories\Reports\OnHandAssetAgeRepository;
use App\Repositories\CommonFilterRepository;
use App\Http\Responses\ReportOutputData;


class OnHandAssetAgeService extends AbstractReportService
{
    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Constructs a new instance of the class.
     *
     * @param OnHandAssetAgeRepository $repository The OnHandAssetAgeRepository instance.
     * @param CommonFilterRepository $commonFilterRepository The commonFilterRepository instance.
     * @param ReportOutputData $reportOutputData The reportOutputData instance
     */
    public function __construct(OnHandAssetAgeRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Retrieves the data for the report.
     *
     * @return array Returns an array containing the filtered assets and the count.
     */
    public function data()
    {
        $currentDate = Carbon::now()->format('Y-m-d');
        $assets = $this->repository->getOnHandAssetAgeAssets($currentDate);
        $requestedData = request()->form;
        $inputData = $this->getInputData($requestedData);
        $assets = $this->filterResults($assets, $inputData);
        $count = $assets->get()->count();
        $assets = $this->reportOutputData->getOutputData($assets, ['assets.created_at' => 'asc']);

        return compact('assets', 'count');
    }

    /**
     * Retrieves the input data for processing.
     *
     * @param array $requestedData The requested data to retrieve.
     * @return array
     */
    public function getInputData($requestedData)
    {
        return [
            'assetType'    => $requestedData['asset_type'] ?? '',
            'makeAndModel' => $requestedData['make_and_model'] ?? '',
            'techSpecs'    => $requestedData['technical_spec'] ?? '',
            'location'     => $requestedData['location'] ?? '',
            'assetStatus'  => $requestedData['status'] ?? '',
            'asset_age'    => $requestedData['asset_age'] ?? '',
        ];
    }

    /**
     * Filters the results based on the input data.
     *
     * @param array $assets The array of assets to filter.
     * @param array $inputData The input data used for filtering.
     * @return object The filtered array of assets.
     */
    public function filterResults($assets, $inputData)
    {
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'assets.asset_type_id', $inputData['assetType']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'assets.make_and_model_id', $inputData['makeAndModel']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'assets.technical_spec_id', $inputData['techSpecs']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'assets.location_id', $inputData['location']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'assets.asset_status_id', $inputData['assetStatus']);
        $assets = $this->commonFilterRepository->filterAssetsByAgeGroup($assets, 'assets.created_at', $inputData['asset_age']);


        return $assets;
    }

    /**
     * Retrieves the export data based on the input data.
     *
     * @return mixed The export data.
     */
    public function exportData()
    {
        $currentDate = Carbon::now()->format('Y-m-d');
        $assets = $this->repository->getOnHandAssetAgeAssets($currentDate);
        $requestedData = request()->all();
        $inputData = $this->getInputData($requestedData);
        $assets = $this->filterResults($assets, $inputData);
        $assets = $assets->orderBy('assets.created_at');

        return $assets;
    }

    /**
     * Returns an array of nested data for exporting.
     *
     * @param mixed $asset The asset object.
     * @return array The nested data array.
     */
    public function getReportNestedData($asset)
    {
        $nestedData['report']['asset_tag']         = $nestedData['export']['Asset Tag #'] = $asset->asset_tag;
        $nestedData['report']['serial_no']         = $nestedData['export']['Serial #'] = $asset->serial_no;
        $nestedData['report']['asset_type']        = $nestedData['export']['Asset Type'] = $asset->assetType ? $asset->assetType->name : '';
        $nestedData['report']['hardware_standard'] = $nestedData['export']['Hardware Standard'] = $asset->makeAndModel ? $asset->makeAndModel->makeModelName : '';
        $nestedData['report']['tech_specs']        = $nestedData['export']['Technical Specs'] = $asset->technicalSpec ? $asset->technicalSpec->details : '';
        $nestedData['report']['location']          = $nestedData['export']['Location'] = $asset->location ? $asset->location->room_name : '';
        $nestedData['report']['asset_status']      = $nestedData['export']['Asset Status'] = $asset->assetStatus ? $asset->assetStatus->name : '';
        $nestedData['report']['asset_age']         = $nestedData['export']['Asset Age'] = $asset->AssetAge;
        $nestedData['report']['created_at']        = $nestedData['export']['Date Created'] = $asset->created_at;

        return $nestedData;
    }

    /**
     * Retrieves nested data for a given asset and index.
     *
     * @param mixed $asset The asset object.
     * @param int $index The index of the nested data.
     *
     * @return array The nested data.
     */
    public function getNestedData($asset, $index)
    {
        $nestedData = [];
        $commonData = $this->getReportNestedData($asset);
        $nestedData = $commonData['report'];
        $nestedData['id']   = $index;
        $nestedData['asset_tag'] = generateAssetLink($asset->id, $asset->asset_tag);
        $nestedData['serial_no'] = generateAssetLink($asset->id, $asset->serial_no);
        $nestedData['location'] = generateLocationLink($asset->location?->id, $asset->location?->room_name);

        return $nestedData;
    }
}
