<?php

namespace App\Services\Reports;

use App\Models\AssetType;
use App\Models\MakeAndModel;
use App\Models\AssetStatus;
use App\Repositories\InStockComputerRepository;
use App\Repositories\CommonFilterRepository;
use App\Http\Responses\ReportOutputData;

/**
 * Service class for InStock Computer Report
 */
class InStockComputerService extends AbstractReportService
{

    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Construct of InStockComputerService
     * @param InStockComputerRepository $repository The repository for the InStockComputer Repository entity
     * @param CommonFilterRepository $commonFilterRepository The repository for the CommonFilter entity
     * @param ReportOutputData $reportOutputData The ReportOutputData entity
     */
    public function __construct(InStockComputerRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Data to the Report data table
     *
     * @return void
     */
    public function data()
    {
        $requestedData  = request()->form;
        $inputData      = $this->getInputData($requestedData);
        $result         = $this->repository->getInstockComputers();
        $result         = $this->filterWithInputData($result, $inputData);
        $count          = $result->count();
        $result         = $this->reportOutputData->getOutputData($result, ['id' => 'asc']);

        return compact('result', 'count');
    }

    /**
     * Filter the query results based on the filters applied
     *
     * @param  mixed $assets
     * @param  mixed $inputData
     * @return object
     */
    public function filterWithInputData($assets, $inputData)
    {
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $inputData['assetType']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_status_id', $inputData['status']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'make_and_model_id', $inputData['makeAndModel']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'location_id', $inputData['location']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'technical_spec_id', $inputData['tech_spec']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'makeAndModel.manufacturer', 'id', $inputData['manufacturer']);

        return $assets;
    }

    /**
     * Taking the filter field data for listing
     *
     * @param array $requestedData
     */
    public function getInputData($requestedData)
    {
        return [
            'assetType'     => $requestedData['asset_type'] ?? '',
            'status'        => $requestedData['status'] ?? '',
            'makeAndModel'  => $requestedData['make_and_model'] ?? '',
            'manufacturer'  => $requestedData['manufacturer'] ?? '',
            'location'      => $requestedData['location'] ?? '',
            'tech_spec'     => $requestedData['tech_spec'] ?? '',
        ];
    }

    /**
     * export Data as csv
     *
     * @return mixed
     */
    public function exportData()
    {
        $requestedData = request()->all();
        $inputData = $this->getInputData($requestedData);
        $assets     = $this->repository->getInstockComputers();
        $assets  = $this->filterWithInputData($assets, $inputData);
        $assets     = $assets->orderBy('id');

        return $assets;
    }


    /**
     * get Report and export common Nested Data
     *
     * @param  mixed $history
     * @return void
     */
    public function getReportNestedData($asset)
    {
        $nestedData['report']['asset_tag'] = $nestedData['export']['Asset Tag #'] = $asset->asset_tag;
        $nestedData['report']['serial_number'] = $nestedData['export']['Serial #'] = $asset->serial_no;
        $nestedData['report']['type'] = $nestedData['export']['Asset Type'] = $nestedData['Type'] =  $asset->assetType ? $asset->assetType->name : '';
        $nestedData['report']['manufacturer'] = $nestedData['export']['Manufacturer'] = optional(optional($asset->makeAndModel)->manufacturer)->name;
        $nestedData['report']['hardware_standard'] = $nestedData['export']['Hardware Standard'] = optional($asset->makeAndModel)->name;
        $nestedData['report']['technical_spec'] = $nestedData['export']['Technical Specs'] = $asset->technicalSpec ? $asset->technicalSpec->details : '';
        $nestedData['report']['po_number'] = $nestedData['export']['PO #'] = $asset->po_id;
        $nestedData['report']['asset_status'] = $nestedData['export']['Asset Status'] = $asset->assetStatus ? $asset->assetStatus->name : '';
        $nestedData['report']['user_location'] = $nestedData['export']['Location'] = $asset->location?->room_name;
        $nestedData['report']['warranty_end_date'] = $nestedData['export']['Warranty End Date'] = $asset->warranty_end_date;
        $nestedData['report']['age'] = $nestedData['export']['Asset Age'] = $asset->AssetAge;
        $nestedData['report']['modified_date'] = $nestedData['export']['Last Modified Date'] = $asset->latestAssetHistory ? $asset->latestAssetHistory->updated_at : '';
        $nestedData['report']['modified_by'] = $nestedData['export']['Last Modified By'] = $asset->latestAssetHistory ? (optional($asset->latestAssetHistory->user)->first_name . ' ' . optional($asset->latestAssetHistory->user)->last_name) : '';

        return $nestedData;
    }

    /**
     * nested data for report
     *
     * @return array
     */
    public function getNestedData($item, $index)
    {
        $nestedData = [];
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id']   = $index;
        $nestedData['asset_tag'] = generateAssetLink($item->id, $item->asset_tag);
        $nestedData['serial_number'] = generateAssetLink($item->id, $item->serial_no);
        $nestedData['user_location'] = generateLocationLink($item->location?->id, $item->location?->room_name);
        $nestedData['modified_by'] = $item->latestAssetHistory ? generateUserLink($item->latestAssetHistory->user_id, $nestedData['modified_by'] . "</a>") : '';

        return $nestedData;
    }

    /**
     * get filter values based on status and type.
     * @return array
     */
    public function getSearchData($status, $type)
    {
        $assetTypeIds = AssetType::whereIn('slug', ['desktop', 'laptop', 'computer'])->pluck('id')->toArray();
        $manufacturerIds = $this->repository->getManufacturerByType($type);
        $makeModels = MakeAndModel::whereIn('manufacturer_id', $manufacturerIds)->whereIn('asset_type_id', $assetTypeIds);

        if ($type == "chromebook") {
            $makeModels = $makeModels->where('slug', 'like', '%chromebook%')->pluck('id')->toArray();
            $makeModels = !empty($makeModels) ? $makeModels : 'null';
        }

        if ($type == "pc") {
            $makeModels = $makeModels->where('slug', 'not like', '%chromebook%')->pluck('id')->toArray();
        }

        return [
            'asset_type'     => $assetTypeIds,
            'asset_status'   => AssetStatus::getStatusID($status)->pluck('id')->toArray(),
            'manufacturer'   => $manufacturerIds,
            'make_and_model' => $makeModels,
            'device_type'    => $type
        ];
    }
}
