<?php

namespace App\Services\Reports;

use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\MakeAndModel;
use App\Models\Manufacturer;
use App\Repositories\CommonFilterRepository;
use Carbon\Carbon;
use App\Repositories\ReportGeneratorRepository;
use Illuminate\Support\Facades\File;

class Generator
{
    protected $repository;
    protected $commonFilterRepository;

    public function __construct(ReportGeneratorRepository $repository, CommonFilterRepository $commonFilterRepository)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
    }

    /**
     * Making query for the report
     */
    public function data()
    {
        $inputData  = $this->repository->getInputData();
        $assets     = $this->repository->reportGeneratorQuery();

        // for filter with asset history date fields
        if (isset($inputData['last_modified_date_from']) || isset($inputData['last_modified_date_to'])) {
            $assets = $this->repository->appendAssetHistoryJoinQuery($assets);
        }
        $assets     = $this->filterAssets($assets, $inputData);
        $count      = $assets->count();
        $assets     = $this->getOutputData($assets, $inputData);

        return compact('assets', 'count');
    }

    /**
     * Making the query for export
     */
    public function exportData($requestData = array())
    {
        if (!empty($requestData)) {
            request()->merge($requestData);
        }

        $inputData  = $this->repository->getExportInputData();
        $assets     = $this->repository->reportGeneratorQuery();

        // for filter with asset history date fields
        if (isset($inputData['last_modified_date_from']) || isset($inputData['last_modified_date_to'])) {
            $assets = $this->repository->appendAssetHistoryJoinQuery($assets);
        }
        $assets  = $this->filterAssets($assets, $inputData);
        $assets->orderBy('id', 'ASC');
        return $assets;
    }

    public function getSavedSearchParameters()
    {
        foreach (config('asset-fields.select') as $key => $fields) {
            if (request($key)) {
                $parameters[$key] = [$fields['db_name'] => request($key)];
            }
        }

        foreach (config('asset-fields.boolean') as $key => $fields) {
            if (request($key)) {
                $parameters[$key] = [$fields['db_name'] => request($key)];
            }
        }

        foreach (config('asset-fields.dates_from') as $key => $fields) {
            if (request($fields)) {
                $parameters[$fields] = [$key => request($fields)];
            }
        }

        foreach (config('asset-fields.dates_to') as $key => $fields) {
            if (request($fields)) {
                $parameters[$fields] = [$key => request($fields)];
            }
        }

        if (request('asset_age_value')) {
            $parameters['asset_age_value'] = ['asset_age_value' => request('asset_age_value')];
        }

        if (request('asset_age_unit')) {
            $parameters['asset_age_unit'] = ['asset_age_unit' => request('asset_age_unit')];
        }

        if (request('depreciated_value_from')) {
            $parameters['depreciated_value_from'] = ['depreciated_value_from' => request('depreciated_value_from')];
        }

        if (request('depreciated_value_to')) {
            $parameters['depreciated_value_to'] = ['depreciated_value_to' => request('depreciated_value_to')];
        }

        foreach (config('asset-fields.radioButton') as $key => $fields) {
            $parameters[$key] = [$fields['db_name'] => request($key)];
        }

        if (request('show_linked')) {
            $parameters['show_linked'] = ['show_linked' => request('show_linked')];
        }
        if (request('show_individual')) {
            $parameters['show_individual'] = ['show_individual' => request('show_individual')];
        }

        return $parameters;
    }

    public function getAssetData($assets, $start, $data)
    {
        foreach ($assets as $key => $asset) {
            $nestedData['id'] = $start + ($key + 1);
            $nestedData['type'] = $asset->assetType ? $asset->assetType->name : '';
            $nestedData['manufacturer'] = optional(optional($asset->makeAndModel)->manufacturer)->name;
            $nestedData['po_number'] = $asset->po_id;
            $nestedData['vendor'] = optional($asset->vendor)->name;

            $nestedData['asset_tag'] = generateAssetLink($asset->id, $asset->asset_tag);
            $nestedData['child_asset'] = $asset->child;
            $nestedData['serial_number'] = generateAssetLink($asset->id, $asset->serial_no);
            $nestedData['hardware_standard'] = $asset->makeAndModel ? $asset->makeAndModel->makeModelName : '';
            $nestedData['technical_spec'] = $asset->technicalSpec ? $asset->technicalSpec->details : '';
            $nestedData['asset_status'] = $asset->assetStatus ? $asset->assetStatus->name : '';
            $nestedData['lock_status'] = $asset->lock_status;
            $nestedData['imei'] = $asset->imei;
            $nestedData['user_location'] = $asset->user ? generateUserLink($asset->user?->id, $asset->user?->user_name) : generateLocationLink($asset->location?->id, $asset->location?->room_name);
            if ($asset->user && $asset->location) {
                $nestedData['user_location'] = generateUserLink($asset->user->id, $asset->user?->user_name) . " / " . generateLocationLink($asset->location?->id, $asset->location?->room_name);
            }
            $nestedData['email'] = generateUserLink($asset->user?->id, $asset->user?->email);
            $nestedData['employee_id'] = $asset->user ? $asset->user->employee_id : '';
            $nestedData['business_title'] = $asset->user ? optional($asset->user->position)->name : '';
            $nestedData['department'] = $asset->user ? optional($asset->user->department)->name : '';
            $nestedData['work_location'] = $asset->user ? $asset->user->city : '';
            $nestedData['legal_hold'] = $asset->user ? ($asset->user->legalhold == 1 ? 'Yes' : 'No') : '';
            if (!empty($asset->user && $asset->user->hire_date != '0000-00-00' && $asset->user->hire_date != '11/30/-0001')) {
                $hireDate = $asset->user->hire_date;
            } else {
                $hireDate = '';
            }
            $nestedData['hire_date'] = $hireDate;
            $nestedData['terminated_date'] = $asset->user ? $asset->user->terminated_date : '';
            $nestedData['status'] = $asset->user ? ($asset->user->status == 1 ? 'Active' : 'Terminated') : '';

            $nestedData['date_deployed'] = $asset->date_deployed;
            $nestedData['lease_start_date'] = $asset->lease_start_date;
            $nestedData['lease_end_date'] = $asset->lease_end_date;
            $nestedData['loaner_return_date'] = $asset->loaner_return_date;
            $nestedData['loaner_retention_date'] = $asset->loaner_retention_date;
            $nestedData['ticket_no'] = $asset->ticket_no;
            $nestedData['carrier'] = $asset->carrier ? $asset->carrier->name : '';
            $nestedData['parent_asset_tag'] = $asset->parentAsset ? $asset->parentAsset->asset_tag : '';
            $nestedData['linked_date'] = $asset->linked_date;
            $nestedData['lost_date'] = $asset->lost_date;
            $nestedData['end_of_life_date'] = $asset->end_of_life_date;
            $nestedData['warranty_end_date'] = $asset->warranty_end_date;
            $nestedData['ewaste_certificate'] = '';

            if ($asset->ewaste_certificate) {
                $nestedData['ewaste_certificate'] = '<a href="/downloadcertificate/' . str_replace('/', '-', $asset->ewaste_certificate) . '">';
                $nestedData['ewaste_certificate'] .= strlen(sanitizeFilename($asset->ewaste_certificate)) > 20 ?
                    "..." . substr(sanitizeFilename($asset->ewaste_certificate), strlen(sanitizeFilename($asset->ewaste_certificate))
                        - 20, strlen(sanitizeFilename($asset->ewaste_certificate))) :
                    sanitizeFilename($asset->ewaste_certificate);
                $nestedData['ewaste_certificate'] .= '</a>';
            }
            $nestedData['age'] = $asset->AssetAge;
            $nestedData['original_value'] = config('currency.currency_symbol') . number_format((float) $asset->original_value) ?? '';
            $nestedData['deprecated_value'] = config('currency.currency_symbol') . number_format((float) $asset->depreciatedValue) ?? '';
            $nestedData['modified_date'] = $asset->latestAssetHistory ? $asset->latestAssetHistory->updated_at : '';
            $nestedData['modified_by'] = generateAssetLink($asset->latestAssetHistory?->user?->id, $asset->latestAssetHistory?->user?->user_name);
            $nestedData['created_at'] = $asset->created_at;

            $data[] = $nestedData;
        }


        return $data;
    }

    /**
     * Making the export data array
     * @param mixed $assets
     * @param mixed $start
     * @param mixed $data
     *
     * @return [type]
     */
    public function getExportAssetData($assets, $start, $data)
    {
        foreach ($assets as $key => $asset) {
            $hireDate = '';
            if (!empty($asset->user && $asset->user->hire_date != '0000-00-00' && $asset->user->hire_date != '11/30/-0001')) {
                $hireDate = $asset->user->hire_date;
            }
            $nestedData['ID'] = $start + ($key + 1);
            $nestedData['Asset Type'] = disableCSVInjection($asset->assetType ? $asset->assetType->name : '');
            $nestedData['Manufacturer'] = disableCSVInjection(optional(optional($asset->makeAndModel)->manufacturer)->name);
            $nestedData['PO #'] = disableCSVInjection($asset->po_id);
            $nestedData['Vendor'] = disableCSVInjection(optional($asset->vendor)->name);

            $nestedData['Asset Tag #'] = disableCSVInjection($asset->asset_tag);
            $nestedData['Child Asset'] = disableCSVInjection($asset->child);
            $nestedData['Serial #'] = disableCSVInjection($asset->serial_no);
            $nestedData['Hardware Standard'] = disableCSVInjection($asset->makeAndModel ? $asset->makeAndModel->makeModelName : '');
            $nestedData['Technical Specs'] = disableCSVInjection($asset->technicalSpec ? $asset->technicalSpec->details : '');
            $nestedData['Asset Status'] = disableCSVInjection($asset->assetStatus ? $asset->assetStatus->name : '');
            $nestedData['Lock Status'] = disableCSVInjection($asset->lock_status);
            $nestedData['IMEI'] = disableCSVInjection($asset->imei);
            $nestedData['User Location'] = disableCSVInjection($asset->user ? ($asset->user->first_name . ' ' . $asset->user->last_name) : ($asset->location ? $asset->location->room_name : ''));

            if ($asset->user && $asset->location) {
                $nestedData['User Location'] = ($asset->user->first_name . ' ' . $asset->user->last_name . ' / ' . $asset->location->room_name);
            }
            $nestedData['Email'] = disableCSVInjection($asset->user ? $asset->user->email : '');
            $nestedData['Employee ID'] = disableCSVInjection($asset->user ? $asset->user->employee_id : '');
            $nestedData['Business Title'] = disableCSVInjection($asset->user ? optional($asset->user->position)->name : '');
            $nestedData['Department'] = disableCSVInjection($asset->user ? optional($asset->user->department)->name : '');
            $nestedData['Work Location'] = disableCSVInjection($asset->user ? $asset->user->city : '');
            $nestedData['Legal Hold'] = disableCSVInjection($asset->user ? ($asset->user->legalhold == 1 ? 'Yes' : 'No') : '');
            $nestedData['Hire Date'] = disableCSVInjection($hireDate);
            $nestedData['Terminated Date'] = disableCSVInjection($asset->user ? $asset->user->terminated_date : '');
            $nestedData['Status'] = disableCSVInjection($asset->user ? ($asset->user->status == 1 ? 'Active' : 'Terminated') : '');

            $nestedData['Date Deployed'] = disableCSVInjection($asset->date_deployed);
            $nestedData['Lease Start Date'] = disableCSVInjection($asset->lease_start_date);
            $nestedData['Lease End Date'] = disableCSVInjection($asset->lease_end_date);
            $nestedData['Expected Return Date'] = disableCSVInjection($asset->loaner_return_date);
            $nestedData['Loaner Retention Date'] = disableCSVInjection($asset->loaner_retention_date);
            $nestedData['Ticket No'] = disableCSVInjection($asset->ticket_no);
            $nestedData['Carrier'] = disableCSVInjection($asset->carrier ? $asset->carrier->name : '');
            $nestedData['Parent Asset Tag #'] = disableCSVInjection($asset->parentAsset ? $asset->parentAsset->asset_tag : '');
            $nestedData['Linked Date'] = disableCSVInjection($asset->linked_date);
            $nestedData['Lost Date'] = disableCSVInjection($asset->lost_date);
            $nestedData['End of Life Date'] = disableCSVInjection($asset->end_of_life_date);
            $eWasteCertificate = $asset->ewaste_certificate ? route('downloadcertificate', ['file' => str_replace('/', '-', $asset->ewaste_certificate)]) : '';
            $nestedData['ITAD Certificate'] = disableCSVInjection($eWasteCertificate);
            $nestedData['Asset Age'] = disableCSVInjection($asset->AssetAge);
            $nestedData['Asset Value'] = disableCSVInjection(optional($asset->technicalSpec)->original_value);
            $nestedData['Depreciation Value'] = disableCSVInjection($asset->depreciatedValue ?? '');
            $nestedData['Created at'] = disableCSVInjection($asset->created_at);
            $nestedData['Warranty End Date'] = disableCSVInjection($asset->warranty_end_date);
            $nestedData['Last Modified Date'] = disableCSVInjection($asset->latestAssetHistory ? $asset->latestAssetHistory->updated_at : '');
            $nestedData['Last Modified By'] = disableCSVInjection($asset->latestAssetHistory ? (optional($asset->latestAssetHistory->user)->first_name . '' . optional($asset->latestAssetHistory->user)->last_name) : '');

            $data[] = $nestedData;
        }
        return $data;
    }


    /**
     * filter queries based on inputs
     * @param mixed $assets
     * @param mixed $inputData
     *
     * @return [type]
     */
    public function filterAssets($assets, $inputData)
    {
        foreach (config('asset-fields.select') as $key => $fields) {
            if (isset($inputData[$key])) {
                if (in_array($key, ['department', 'employee_type', 'work_location'])) {
                    $assets = $this->repository->filterWithUserFields($assets,  $fields['db_name'], $inputData[$key]);
                    continue;
                }
                if (in_array($key, ['site_code', 'building'])) {
                    $assets = $this->repository->filterWithLocationFields($assets, $fields['db_name'], $inputData[$key]);
                    continue;
                }
                if (in_array($key, ['manufacturer'])) {
                    $assets = $this->repository->filterWithMakeAndModelFields($assets, $fields['db_name'], $inputData[$key]);
                    continue;
                }
                if (in_array($key, ['original_value'])) {
                    $assets = $this->repository->filterWithAssetValue($assets, $inputData[$key]);
                    continue;
                }

                if (in_array($key, ['depreciated_value'])) {
                    $assets = $this->repository->filterWithDepreciatedValue($assets, $inputData[$key]);
                    continue;
                }

                $assets = $this->repository->filter($assets, $fields['db_name'], $inputData[$key]);
            }
        }

        foreach (config('asset-fields.boolean') as $key => $fields) {
            if (isset($inputData[$key])) {
                if ($inputData[$key] == 1) {
                    if ($key == 'have_assets') {
                        $assets = $assets->whereNotNull('user_id');
                    } elseif ($key == 'legalhold') {
                        $assets = $assets->whereHas('user', function ($query) {
                            $query->legalHold();
                        });
                    } else {
                        $assets = $assets->whereNotNull($fields['db_name']);
                    }
                }
                if ($inputData[$key] == 2) {
                    if ($key == 'have_assets') {
                        $assets = $assets->whereNull('user_id');
                    } elseif ($key == 'legalhold') {
                        $assets = $assets->whereHas('user', function ($query) {
                            $query->notLegalHold();
                        });
                    } else {
                        $assets = $assets->whereNull($fields['db_name']);
                    }
                }
            }
        }

        foreach (config('asset-fields.dates_from') as $key => $fields) {
            if (isset($inputData[$fields])) {
                if ($key == 'hire_date') {
                    $assets = $assets->whereHas('user', function ($query) use ($key, $fields, $inputData) {
                        $query->where($key, '>=', convert_to_db_date($inputData[$fields]));
                    });
                } elseif ($key == 'last_modified_date') {
                    $assets->whereDate('asset_histories.updated_at', '>=', convert_to_db_date($inputData[$fields]));
                } else {
                    $assets = $assets->where("assets." . $key, '>=', convert_to_db_date($inputData[$fields]));
                }
            }
        }

        foreach (config('asset-fields.dates_to') as $key => $fields) {
            if (isset($inputData[$fields])) {
                if ($key == 'hire_date') {
                    $assets = $assets->whereHas('user', function ($query) use ($key, $fields, $inputData) {
                        $query->where($key, '<=', convert_to_db_date($inputData[$fields]));
                    });
                } elseif ($key == 'last_modified_date') {
                    $assets->whereDate('asset_histories.updated_at', '<=', convert_to_db_date($inputData[$fields]));
                } else {
                    $assets = $assets->where("assets." . $key, '<=', convert_to_db_date($inputData[$fields]));
                }
            }
        }

        foreach (config('asset-fields.radioButton') as $key => $fields) {
            if (isset($inputData[$key])) {
                if ($inputData[$key] == 1) {
                    $assets = $assets->whereHas($fields['relation'], function ($query) use ($key, $fields, $inputData) {
                        if ($fields['relation'] == 'user') {
                            $query->where('status', 1);;
                        } else {
                            $query->where($fields['db_name'], 1);
                        }
                    });
                }
                if ($inputData[$key] == 2) {
                    $assets = $assets->whereHas($fields['relation'], function ($query) use ($key, $fields, $inputData) {
                        if ($fields['relation'] == 'user') {
                            $query->whereNull('status')->orWhere('status', '!=', 1);
                        } else {
                            $query->where($fields['db_name'], 0);
                        }
                    });
                }
            }
        }

        if (isset($inputData['show_linked'])) {
            if ($inputData['show_linked'] == 1) {
                $assets = $assets->where(function ($assets) {
                    $assets->has('childrenAsset')->orHas('parentAsset');
                });
            }
        }

        if (isset($inputData['show_individual'])) {
            if ($inputData['show_individual'] != 1) {
                $assets = $assets->whereNull('parent_asset_id');
            }
        } else {
            $assets = $assets->whereNull('parent_asset_id');
        }

        if ($inputData['asset_age_value']) {
            $date = $this->calculateDate($inputData['asset_age_value'], $inputData['asset_age_unit']);
            $assets = $assets->where('assets.created_at', '<=', $date);
        }

        // if (isset($inputData['depreciated_value_from']) && $inputData['depreciated_value_from']) {
        //     // $assets->where('depreciated_value_percentage', '>=', $inputData['depreciated_value_from']);
        //     $assets = $assets->get()->filter(function ($item, $inputData) {
        //         return $item->depreciatedValuePercentage >= $inputData['depreciated_value_from'];
        //     })->all();
        // }

        // if (isset($inputData['depreciated_value_to']) && $inputData['depreciated_value_to']) {
        //     // $assets->where('depreciated_value_percentage', '<=', $inputData['depreciated_value_to']);
        //     $assets = $assets->get()->filter(function ($item, $inputData) {
        //         return $item->depreciatedValuePercentage <= $inputData['depreciated_value_to'];
        //     })->all();
        // }

        return $assets;
    }

    /**
     * Executing the query
     * @param $users Query
     * @return Query Result
     */
    public function getOutputData($assets, $inputData)
    {
        $start = request('start');
        $limit = request('length');

        if ($limit != -1) {
            $assets = $assets->offset($start)
                ->limit($limit);
        }
        $assets->orderBy('assets.id', 'ASC');

        return $assets->get();
    }

    public function calculatePercentage($item, $depreciatedFrom)
    {
        if ($item->technicalSpec && (int)$item->technicalSpec->original_value != 0) {
            return round(((int)$item->depreciatedValue / (int)$item->technicalSpec->original_value) * 100, 2);
        }
        return 100;
    }

    public function calculateDate($assetAge, $ageUnit = '')
    {
        if ($ageUnit == 'day') {
            $date = Carbon::now()->subDays($assetAge);
        }

        if ($ageUnit == 'month') {
            $date = Carbon::now()->subMonths($assetAge);
        }

        if ($ageUnit == 'year') {
            $date = Carbon::now()->subYears($assetAge);
        }

        return $date;
    }

    public function decodeSearch($parameters)
    {
        $search = collect(json_decode($parameters))->toArray();
        $search = array_map(function ($object) {
            return (array) $object;
        }, $search);

        return $search;
    }

    /**
     * storing export file
     */
    public function saveExport($datas, $title)
    {
        $fileName  = str_slug($title, '_') . '_' . time();
        $return['file'] = $fileName;
        $return['title'] = $title;

        if (!File::isDirectory(storage_path("app/public/share_via_email"))) {
            File::makeDirectory(storage_path("app/public/share_via_email"), 0755);
        }


        $filePath  = storage_path("app/public/share_via_email/" . $fileName . '.csv');

        $handle = fopen($filePath, 'w');

        if (!array_key_exists(0, $datas[0])) {
            fputcsv($handle, ['No data available']);
            fclose($handle);

            return response()->json($return);
        }

        fputcsv($handle, array_keys($datas[0][0]));
        foreach ($datas as $data) {
            foreach ($data as $data) {
                fputcsv($handle, $data);
            }
        }
        fclose($handle);

        return response()->json($return);
    }

    public function getSearch($status, $type, $hardwarStandard = null, $department = null)
    {
        $search = [];
        $dateFrom = Carbon::now()->startOfWeek()->format('m/d/Y');
        $dateTo = Carbon::now()->endOfWeek()->format('m/d/Y');

        $assetStatus = AssetStatus::getStatusID($status)->first();
        if ($assetStatus) {
            $assetStatusId = $assetStatus->id;
            $search['asset_status_id'] = ['asset_status_id' => $assetStatusId];
        }

        if ($status == 'loaned') {
            if ($type == 'due_this_week') {
                $search['loaner_return_date_from'] = ['loaner_return_date' => $dateFrom];
                $search['loaner_return_date_to'] = ['loaner_return_date' => $dateTo];
            }

            if ($type == 'past_due') {
                $search['loaner_return_date_to'] = ['loaner_return_date' => $dateFrom];
            }
        }

        if ($status == 'retain_hold') {
            if ($type == 'due_this_week') {
                $search['loaner_retention_date_from'] = ['loaner_retention_date' => $dateFrom];
                $search['loaner_retention_date_to'] = ['loaner_retention_date' => $dateTo];
            }

            if ($type == 'past_due') {
                $search['loaner_retention_date_to'] = ['loaner_retention_date' => $dateFrom];
            }
        }

        if ($status == 'lease') {
            if ($type == 'due_this_week') {
                $search['lease_end_date_from'] = ['lease_end_date' => $dateFrom];
                $search['lease_end_date_to'] = ['lease_end_date' => $dateTo];
            }

            if ($type == 'past_due') {
                $search['lease_end_date_to'] = ['lease_end_date' => $dateFrom];
            }
        }

        if ($status == 'usedandnew') {
            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusIdsBySlug(['brand_new', 'refurb_ready_to_deploy', 'used', 'like_new', 'refurb'])->pluck('id')->toArray()
            ];

            //$locations = Location::where('region_id', $type)->pluck('id')->toArray();
            $search['location'] = [
                'location_id' => $type
            ];

            if ($hardwarStandard != '') {
                $search['make_and_model'] = [
                    'make_and_model_id' => $hardwarStandard
                ];
            }

            $search['asset_type'] = [
                'asset_type_id' => AssetType::getBySlugList(['computer', 'desktop', 'laptop'])->pluck('id')->toArray()
            ];
        }

        if ($status == 'asset_type') {
            $assetTypeId = AssetType::getBySlug($type)->first()->id;
            $search['asset_type'] = [
                'asset_type_id' => $assetTypeId
            ];
        }

        if ($status == 'inactive_users') {
            $search['user_status'] = [
                'status' => 2
            ];

            $search['asset_status_id'] = [
                'asset_status_id' => [
                    AssetStatus::getStatusID('assigned')->first()->id,
                    AssetStatus::getStatusID('loaned')->first()->id
                ]
            ];
        }

        if ($status == 'inactive_locations') {
            $search['location_status'] = [
                'status' => 2
            ];
        }

        if ($status == 'ageover_fouryears') {
            $search['asset_age_value'] = [
                'asset_age_value' => 4,
            ];
            $search['asset_age_unit'] = [
                'asset_age_unit' => 'year'
            ];
        }


        if ($status == 'ageover_threeyears') {
            $search['asset_age_value'] = [
                'asset_age_value' => 3,
            ];
            $search['asset_age_unit'] = [
                'asset_age_unit' => 'year'
            ];

            $search['asset_status_id'] = [
                'asset_status_id' => [
                    AssetStatus::getStatusID('assigned')->first()->id
                ]
            ];

            $search['asset_type'] = [
                'asset_type_id' => AssetType::getBySlugList(['computer', 'desktop', 'laptop'])->pluck('id')->toArray()
            ];

            if ($department != '') {
                $search['department'] = [
                    'department_id' => $department
                ];
            }
        }

        if ($status == 'damagedover_thirtydays') {
            $dateTo =  Carbon::now()->subDays(30)->format('m/d/Y');
            $search['updated_at'] = [
                $search['updated_at_to'] = ['updated_at' => $dateTo]
            ];
            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusID('damaged')->first()->id
            ];
        }

        if ($type == 'without_data_wipe') {
            $search['wipe_confirmation'] = [
                'wipe_confirmation' => 2
            ];
        }

        if ($type == 'without_certificate') {
            $search['end_of_life_data_wipe_certificate'] = [
                'donation_certificate' => 2
            ];
        }

        if ($type == 'computer') {
            $search['asset_type'] = [
                'asset_type_id' => AssetType::getBySlugList(['computer', 'desktop', 'laptop'])->pluck('id')->toArray()
            ];
        }

        if ($type == "mac_computers") {
            $manufacturerId = Manufacturer::whereSlug('apple')->first()->id;

            $assetTypeIds = AssetType::getBySlugList(['computer', 'desktop', 'laptop'])->pluck('id')->toArray();

            $search['asset_type'] = [
                'asset_type_id' => $assetTypeIds
            ];

            $search['make_and_model'] = [
                'make_and_model_id' => MakeAndModel::withoutName(['iPad', 'iPad Air', 'iPad Pro', 'Mac Mini', 'Apple iPad', 'Apple iPod Touch', 'Apple iPad Air', 'Apple iPad Pro', 'Apple iPad Mini 4', 'iPod Touch'])->where('manufacturer_id', $manufacturerId)->whereIn('asset_type_id', $assetTypeIds)->pluck('id')->toArray()
            ];

            $search['manufacturer'] = [
                'manufacturer_id' => $manufacturerId
            ];

            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusIdsBySlug(['brand_new'])->pluck('id')->toArray()
            ];
        }

        if ($type == "mac_refurb_computers") {
            $manufacturerId = Manufacturer::whereSlug('apple')->first()->id;

            $assetTypeIds = AssetType::getBySlugList(['computer', 'desktop', 'laptop'])->pluck('id')->toArray();

            $search['asset_type'] = [
                'asset_type_id' => $assetTypeIds
            ];

            $search['make_and_model'] = [
                'make_and_model_id' => MakeAndModel::withoutName(['iPad', 'iPad Air', 'iPad Pro', 'Mac Mini', 'Apple iPad', 'Apple iPod Touch', 'Apple iPad Air', 'Apple iPad Pro', 'Apple iPad Mini 4', 'iPod Touch'])->where('manufacturer_id', $manufacturerId)->whereIn('asset_type_id', $assetTypeIds)->pluck('id')->toArray()
            ];

            $search['manufacturer'] = [
                'manufacturer_id' => $manufacturerId
            ];

            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusIdsBySlug(['refurb_ready_to_deploy'])->pluck('id')->toArray()
            ];
        }

        if ($type == "refurb_pc") {
            $manufacturerIds = Manufacturer::whereIn('slug', ['dell', 'lenovo'])->pluck('id')->toArray();

            $assetTypeIds = AssetType::getBySlugList(['computer', 'desktop', 'laptop'])->pluck('id')->toArray();

            $search['asset_type'] = [
                'asset_type_id' => $assetTypeIds
            ];

            $search['make_and_model'] = [
                'make_and_model_id' => MakeAndModel::withoutName(['Dell Optiplex 7060', 'Dell Optiplex 7070', 'OptiPlex 5060'])->whereIn('manufacturer_id', $manufacturerIds)->whereIn('asset_type_id', $assetTypeIds)->pluck('id')->toArray()
            ];

            $search['manufacturer'] = [
                'manufacturer_id' => $manufacturerIds
            ];

            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusIdsBySlug(['refurb_ready_to_deploy'])->pluck('id')->toArray()
            ];
        }

        if ($type == "pc") {
            $manufacturerIds = Manufacturer::whereIn('slug', ['dell', 'lenovo'])->pluck('id')->toArray();

            $assetTypeIds = AssetType::getBySlugList(['computer', 'desktop', 'laptop'])->pluck('id')->toArray();

            $search['asset_type'] = [
                'asset_type_id' => $assetTypeIds
            ];

            $search['make_and_model'] = [
                'make_and_model_id' => MakeAndModel::withoutName(['Dell Optiplex 7060', 'Dell Optiplex 7070', 'OptiPlex 5060'])->whereIn('manufacturer_id', $manufacturerIds)->whereIn('asset_type_id', $assetTypeIds)->pluck('id')->toArray()
            ];

            $search['manufacturer'] = [
                'manufacturer_id' => $manufacturerIds
            ];

            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusIdsBySlug(['brand_new'])->pluck('id')->toArray()
            ];
        }

        return $search;
    }

    /**
     * Filters the given assets array with the deprecated value.
     *
     * @param mixed $assets The assets array to be filtered.
     * @param mixed $value The deprecated value used for filtering.
     * @return mixed The filtered assets array.
     */
    public function filterWithDepreciatedValue($assets, $value)
    {
        if ($value == '' || $value == []) {
            return $assets;
        }

        $assets = $assets->filter(function ($item) use ($value) {
            if (is_array($value)) {
                return in_array($item->depreciatedValue, $value);
            }
            return $item->depreciatedValue == $value;
        });

        return $assets;
    }
}
