<?php

namespace App\Services\Reports;

use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Department;
use App\Models\EmployeeType;
use App\Models\Location;
use App\Models\MakeAndModel;
use App\Models\Region;
use Illuminate\Support\Facades\DB;
use App\Repositories\Reports\FleetAgeRepository;
use App\Repositories\CommonFilterRepository;

class FleetAgeService
{

    protected $repository;
    protected $commonFilterRepository;

    /**
     * Constructs a new instance of the class.
     *
     * @param FleetAgeRepository $repository The fleet age repository.
     * @param CommonFilterRepository $commonFilterRepository The common filter repository.
     */
    public function __construct(FleetAgeRepository $repository, CommonFilterRepository $commonFilterRepository)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
    }


    /**
     * Retrieves the filter data options for the application.
     *
     * @return array An array containing the filter data options.
     */
    public function getFilterDataOptions()
    {
        $filterDataOptions['assetTypesAll'] = AssetType::withoutSlug(['accessories']);
        $filterDataOptions['employeeTypes'] = EmployeeType::get();
        $filterDataOptions['regions'] = Region::get();
        $filterDataOptions['hardwareStandards'] = MakeAndModel::with('manufacturer')->get();
        $filterDataOptions['statuses'] = AssetStatus::orderBy('slug')->get();
        $filterDataOptions['locations'] = Location::get();
        $filterDataOptions['departments'] = Department::get();

        return $filterDataOptions;
    }

    /**
     * Retrieves the age and month data.
     *
     * @return array The age and month data.
     */
    public function getAgeandMonthData()
    {
        $data['age'] = 2;

        if (request()->has('selected_age')) {
            $data['age'] = request('selected_age');
        }
        $dateTo = first_day_current_month_future_year($data['age']);
        $dateFrom = first_day_current_month_past_year($data['age']);
        $data['months'] = get_month_year_names($dateFrom, $dateTo);

        return $data;
    }

    /**
     * Retrieves data for the report and returns an array of formatted data.
     *
     * @return array An array of formatted data.
     */
    public function data()
    {
        $ageAndMonthData = $this->getAgeandMonthData();
        $months = $ageAndMonthData['months'];
        $age = $ageAndMonthData['age'];
        $formattedData = [];

        foreach ($months as $monthNumber => $monthName) {
            $monthData = explode('-', $monthNumber);
            $assets = $this->repository->getFleetAgeData();
            $inputData = $this->getInputData();
            $assets = $this->filterWithInputData($assets, $inputData);
            $formattedData[] = $assets->whereMonth(DB::raw('DATE_ADD(created_at, INTERVAL '.$age.' YEAR)'), $monthData[0])->whereYear(DB::raw('DATE_ADD(created_at, INTERVAL '.$age.' YEAR)'), $monthData[1])->count();
        }

        return $formattedData;
    }

    /**
     *  Retrieves the input data for the function.
     *  @return array
     */
    public function getInputData()
    {
        return [
            'selected_asset_type'       =>  request('selected_asset_type') ?? '',
            'selected_status'           =>  request('selected_status') ?? '',
            'selected_standard'         =>  request('selected_standard') ?? '',
            'selected_employee_status'  =>  request('selected_employee_status') ?? '',
            'selected_employee_type'    =>  request('selected_employee_type') ?? '',
            'selected_region'           =>  request('selected_region') ?? '',
            'selected_region'           =>  request('selected_region') ?? '',
            'selected_location'         =>  request('selected_location') ?? '',
            'selected_department'       =>  request('selected_department') ?? '',
        ];
    }
    
    /**
     * Making the query for input filters.
     * 
     * @param object $assets
     * @param array $inputData
     * 
     * @return object
     */
    public function filterWithInputData($assets, $inputData)
    {
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $inputData['selected_asset_type']);    
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'assetStatus', 'slug', $inputData['selected_status']);    
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'make_and_model_id', $inputData['selected_standard']);
        
        if ($inputData['selected_employee_status']) {
            $employeeStatus = $inputData['selected_employee_status'];
            $assets = $assets->whereHas('user', function ($query) use($employeeStatus) {
                if ($employeeStatus == '2') {
                    $query->whereNotNull('terminated_date');
                }
                if ($employeeStatus == '1') {
                    $query->whereNull('terminated_date');
                }
            });
        }
        $assets = $this->commonFilterRepository->filterWithNestedWhereHasRelationFields($assets, 'user', 'employeeType', 'slug', $inputData['selected_employee_type']); 
        $assets = $this->commonFilterRepository->filterWithNestedWhereHasRelationFields($assets, 'location', 'region', 'id', $inputData['selected_region']); 
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'location_id', $inputData['selected_location']);
        $assets = $this->commonFilterRepository->filterWithNestedWhereHasRelationFields($assets, 'user', 'department', 'id', $inputData['selected_department']); 

        return $assets;
    }
}