<?php

namespace App\Services\Reports;

use App\Events\BulkUpdates;
use App\Models\Asset;
use App\Repositories\Reports\DuplicateAssetsRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Http\Responses\ReportOutputData;

class DuplicateAssetsService
{

    protected $repository;
    protected $reportOutputData;

    /**
     * Constructor for the Duplicated Assets service 
     * 
     * @param DuplicateAssetsRepository $repository
     * @param ReportOutputData $reportOutputData
     */
    public function __construct(DuplicateAssetsRepository $repository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Fetching duplicated assets data for report
     * 
     * @return array
     */
    public function getDuplicatedAssetsData()
    {
        $inputData  = $this->getInputData();

        if ($inputData['duplicated_field'] == 'asset_tag') {
            $duplicatedAssets = $this->repository->getAssetsListWithMatchingAssetTag();
        } else {
            $duplicatedAssets = $this->repository->getAssetsListWithMatchingSerialNumber();
        }
        $duplicatedAssets = $this->filterResults($duplicatedAssets, $inputData);
        $count = $duplicatedAssets->count();

        $duplicatedAssets = $this->reportOutputData->getOutputData($duplicatedAssets, ['id' => 'desc']);

        return compact('duplicatedAssets', 'count');
    }

    /**
     * Format assets duplication data for listing
     * 
     * @param object $duplicatedAssets
     * @param int $index
     * @param array $data
     * 
     * @return array 
     */
    public function formatExportAssetDuplicationData($duplicatedAssets, $childAssets)
    {
        $data = [];
        $parentCount = 0;

        $duplicatedChildAssets = $this->repository->getDuplicatedChilds($childAssets);

        foreach ($duplicatedAssets as $duplicatedAssetParent) {
            $parentCount++;
            $nestedData = [];
            $nestedData['#'] = $parentCount;
            $nestedData['Serial #'] = $duplicatedAssetParent->serial_no;
            $nestedData['Asset Tag #'] = $duplicatedAssetParent->asset_tag;
            $nestedData['Asset Type'] = optional($duplicatedAssetParent)->asset_type_name;
            $nestedData['Hardware Standard'] = optional($duplicatedAssetParent)->make_and_model_name;
            $nestedData['Technical Specs'] = optional($duplicatedAssetParent)->technical_spec_details;
            $nestedData['Asset Status'] = optional($duplicatedAssetParent)->asset_status_name;
            $nestedData['User/Location'] = $this->setAssetUserLocation($duplicatedAssetParent, 'export');
            $nestedData['Last Seen'] = $duplicatedAssetParent->getAttribute('lastSeen');
            array_push($data, $nestedData);

            $duplicatedChildrenForThisParent = explode(',', $duplicatedAssetParent->matching_assets);

            $parentIndex = array_search($duplicatedAssetParent->id, $duplicatedChildrenForThisParent);
            unset($duplicatedChildrenForThisParent[$parentIndex]);

            $duplicatedChildrenAssetsOfThisParent = $duplicatedChildAssets->whereIn('id', $duplicatedChildrenForThisParent);

            if (!empty($duplicatedChildrenAssetsOfThisParent)) {
                $childcount = 0;

                foreach ($duplicatedChildrenAssetsOfThisParent as $duplicatedAssetChild) {
                    $childcount++;
                    $nestedData = [];
                    $nestedData['#'] = $parentCount . '.' . $childcount;
                    $nestedData['Serial #'] = $duplicatedAssetChild->serial_no;
                    $nestedData['Asset Tag #'] = $duplicatedAssetChild->asset_tag;
                    $nestedData['Asset Type'] = optional($duplicatedAssetChild->makeAndModel)->name;
                    $nestedData['Hardware Standard'] = optional($duplicatedAssetChild->assetType)->name;
                    $nestedData['Tech Specs'] = optional($duplicatedAssetChild->technicalSpec)->details;
                    $nestedData['Asset Status'] = optional($duplicatedAssetChild->assetStatus)->name;
                    $nestedData['User/Location'] = $this->setAssetUserLocationForChild($duplicatedAssetChild, 'export');
                    $nestedData['Last Seen'] = $duplicatedAssetChild->getAttribute('lastSeen');
                    array_push($data, $nestedData);

                    if ($childcount == $duplicatedChildrenAssetsOfThisParent->count()) {
                        $nestedData = [];
                        $nestedData['Serial #'] = '';
                        $nestedData['Asset Tag #'] = '';
                        $nestedData['Asset Status'] = '';
                        $nestedData['User/Location'] = '';
                        array_push($data, $nestedData);
                    }
                }
            }
        }


        return $data;
    }

    /**
     * Taking the input data
     * 
     * @param Http\Request
     * 
     * @return Array
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();

        return [
            'serialNo'          => $requestedData['serial_no'] ?? '',
            'assetTag'          => $requestedData['asset_tag'] ?? '',
            'user'              => $requestedData['user'] ?? '',
            'assetStatus'       => $requestedData['asset_status'] ?? '',
            'location'          => $requestedData['location'] ?? '',
            'duplicated_field'  => $requestedData['duplicated_field'] ?? '',
        ];
    }

    /**
     * Filer the results based on filters
     * 
     * @param object $duplicatedAssets
     * @param Array $inputData 
     * 
     * @return object
     */
    public function filterResults($duplicatedAssets, $inputData)
    {
        $assets     = $this->repository->filterWithDuplicatedSerialOrTag($duplicatedAssets, 'serial_no', $inputData['serialNo']);
        $assets     = $this->repository->filterWithDuplicatedSerialOrTag($duplicatedAssets, 'asset_tag', $inputData['assetTag']);
        // $assets     = $this->repository->filterWithDuplicatedFields($duplicatedAssets, 'user_id', $inputData['user']);
        // $assets     = $this->repository->filterWithDuplicatedFields($duplicatedAssets, 'matching_statuses', $inputData['assetStatus']);
        // $assets     = $this->repository->filterWithDuplicatedFields($duplicatedAssets, 'location_id', $inputData['location']);

        return $assets;
    }

    /**
     * Fetching duplicated assets data for report
     * 
     * @return object
     */
    public function getDuplicatedAssetsExportData($requestData = array())
    {
        if (!empty($requestData)) {
            request()->merge($requestData);
        }
        $inputData  = $this->getInputData();

        if ($inputData['duplicated_field'] == 'asset_tag') {
            $duplicatedAssets = $this->repository->getAssetsListWithMatchingAssetTag();
        } else {
            $duplicatedAssets = $this->repository->getAssetsListWithMatchingSerialNumber();
        }
        $duplicatedAssets = $this->filterResults($duplicatedAssets, $inputData);
        $duplicatedAssets = $duplicatedAssets->orderBy('id');
        $duplicatedAssets = $duplicatedAssets->get();

        return $duplicatedAssets;
    }

    /**
     * Format assets duplication data for listing
     * 
     * @param object $duplicatedAssets
     * @param int $index
     * @param array $data
     * 
     * @return array
     */
    public function formatAssetDuplicationData($duplicatedAssets, $start, $data, $childAssets, $legthPerPage)
    {
        $duplicatedChildAssets = $this->repository->getDuplicatedChilds($childAssets);
        $countStart = 0;

        foreach ($duplicatedAssets as $duplicatedAssetParent) {
            $countStart++;
            $nestedData                   = [];
            $nestedData['id']             = $start + $countStart;
            $nestedData['serial_no']      = '<a target="_blank" rel="noopener" href="/assets/' . $duplicatedAssetParent->id . '">' . $duplicatedAssetParent->serial_no . '</a>';
            $nestedData['asset_tag']      = '<a target="_blank" rel="noopener" href="/assets/' . $duplicatedAssetParent->id . '">' . $duplicatedAssetParent->asset_tag . '</a>';
            $nestedData['asset_type']     = optional($duplicatedAssetParent)->asset_type_name;
            $nestedData['make_and_model'] = optional($duplicatedAssetParent)->make_and_model_name;
            $nestedData['tech_specs']     = optional($duplicatedAssetParent)->technical_spec_details;
            $nestedData['dataId']         = $duplicatedAssetParent->id;
            $nestedData['asset_status']   = optional($duplicatedAssetParent)->asset_status_name;
            $nestedData['user_location']  = $this->setAssetUserLocation($duplicatedAssetParent);
            $nestedData['last_seen']      = $duplicatedAssetParent->getAttribute('lastSeen');
            $nestedData['dataParent']     = '';
            $nestedData['action']         = $this->getActionButtons($duplicatedAssetParent);
            array_push($data, $nestedData);
            $duplicatedChildrenForThisParent = explode(',', $duplicatedAssetParent->matching_assets);
            $parentIndex = array_search($duplicatedAssetParent->id, $duplicatedChildrenForThisParent);
            unset($duplicatedChildrenForThisParent[$parentIndex]);
            $duplicatedChildrenAssetsOfThisParent = $duplicatedChildAssets->whereIn('id', $duplicatedChildrenForThisParent);

            if (!empty($duplicatedChildrenAssetsOfThisParent)) {
                $childcount = 0;

                foreach ($duplicatedChildrenAssetsOfThisParent as $duplicatedAssetChild) {
                    $childcount++;
                    $nestedData                   = [];
                    $nestedData['id']             = $start + $countStart . '.' . $childcount;
                    $nestedData['serial_no']      = '<a target="_blank" rel="noopener" href="/assets/' . $duplicatedAssetChild->id . '">' . $duplicatedAssetChild->serial_no . '</a>';
                    $nestedData['asset_tag']      = '<a target="_blank" rel="noopener" href="/assets/' . $duplicatedAssetChild->id . '">' . $duplicatedAssetChild->asset_tag . '</a>';
                    $nestedData['make_and_model'] = optional($duplicatedAssetChild->makeAndModel)->name;
                    $nestedData['asset_type']     = optional($duplicatedAssetChild->assetType)->name;
                    $nestedData['tech_specs']     = optional($duplicatedAssetChild->technicalSpec)->details;
                    $nestedData['dataId']         = $duplicatedAssetChild->id;
                    $nestedData['asset_status']   = optional($duplicatedAssetChild->assetStatus)->name;
                    $nestedData['user_location']  = $this->setAssetUserLocationForChild($duplicatedAssetChild);
                    $nestedData['last_seen']      = $duplicatedAssetChild->getAttribute('lastSeen');
                    $nestedData['dataParent']     = $duplicatedAssetParent->id;
                    $isLast                       = ($childcount == count($duplicatedChildrenForThisParent)) ? 'last' : '';
                    $nestedData['dataLast']       = $isLast;
                    $nestedData['action']         = $this->getActionButtonsForChild($duplicatedAssetChild);
                    array_push($data, $nestedData);
                }
            }
        }

        return $data;
    }

    /**
     * Remove duplicated asset
     * 
     * @param int $assetId
     * 
     * @return bool
     */
    public function removeDuplicatedAsset($assetId)
    {
        try {
            $this->repository->deleteDuplicatedAsset($assetId);

            return true;
        } catch (Exception $e) {
            Log::error($e->getMessage());

            return false;
        }
    }

    /**
     * set user location
     * 
     * @param object $asset
     * @param object $action
     * 
     * @return string
     */
    public function setAssetUserLocation($asset, $action = '')
    {
        if (!$asset) {
            return '';
        }

        if ($action == 'export') {
            $userLocation = $asset->location_room_name ? $asset->location_room_name : ($asset->user_first_name . ' ' . $asset->user_last_name);

            if ($asset->user_first_name && $asset->location_room_name) {
                $userLocation = ($asset->user_first_name . ' ' . $asset->user_last_name . ' / ' . $asset->location_room_name);
            }
        } else {
            $userLocation = $asset->location_room_name ? generateLocationLink($asset->location_id, $asset->location_room_name) : generateUserLink($asset->user_id, $asset->user_first_name . ' ' . $asset->user_last_name);

            if ($asset->user_first_name && $asset->location_room_name) {
                $userLocation = generateUserLink($asset->user_id, $asset->user_first_name . ' ' . $asset->user_last_name) . ' / ' . generateLocationLink($asset->location_id, $asset->location_room_name);
            }
        }



        return $userLocation;
    }

    /**
     * set user location
     * 
     * @param object $asset
     * @param object $action
     * 
     * @return string
     */
    public function setAssetUserLocationForChild($asset, $action = '')
    {
        if (!$asset) {
            return '';
        }

        if ($action == 'export') {
            $userLocation = $asset->user ? ($asset->user->first_name . ' ' . $asset->user->last_name) : ($asset->location ? $asset->location->room_name : '');

            if ($asset->user && $asset->location) {
                $userLocation = ($asset->user->first_name . ' ' . $asset->user->last_name . ' / ' . $asset->location->room_name);
            }
        } else {
            $userLocation = $asset->user ?   generateUserLink($asset->user_id, $asset->user_first_name . ' ' . $asset->user_last_name) : ($asset->location ? generateLocationLink($asset->location->id, $asset->location->room_name) : '');

            if ($asset->user && $asset->location) {
                $userLocation = generateUserLink($asset->user_id, $asset->user_first_name . ' ' . $asset->user_last_name) . ' / ' . generateLocationLink($asset->location->id, $asset->location->room_name);
            }
        }

        return $userLocation;
    }

    /**
     * Fetching duplicated assets data for merge
     * 
     * @return object
     */
    public function getDuplicatedAssetsDataForMerge()
    {
        $asset = Asset::where('id', request('id'))->first();
        $duplicatedChildAssets = $this->repository->getDuplicatedChilds(explode(',', request('child_ids')));

        return response()->json(['asset' => $asset, 'duplicatedAssets' => $duplicatedChildAssets]);
    }

    /**
     * Merge duplicate assets
     * 
     * @param int $primaryAssetId
     * @param int $secondaryAssetId
     * 
     * @return array
     */
    public function mergeDuplicatedAssets($primaryAssetId, $secondaryAssetId)
    {
        $response = [];
        $secondaryAssetData = $this->repository->getAssetData($secondaryAssetId);
        $primaryAssetData = $this->repository->getAssetData($primaryAssetId);


        if (!empty($response = $this->checkMatchBeforeMerge($secondaryAssetData, $primaryAssetData))) {
            return $response;
        }
        // $this->repository->mergeDuplicateDataToPrimary($secondaryAssetData, $primaryAssetId);

        $this->repository->mergeDuplicateAssetHistoryToPrimary($primaryAssetData, $secondaryAssetData);
        $this->repository->deleteDuplicatedAsset($secondaryAssetId);

        $response['status'] = 'success';
        $response['message'] = 'The duplicated assets merged to the orginal assets and deleted successfully';

        return $response;
    }

    /**
     * Prevent asset merge if the asset status/user/location fields are different
     * @param mixed $secondaryAssetData
     * @param mixed $primaryAssetData
     * 
     * @return [type]
     */
    private function checkMatchBeforeMerge($secondaryAssetData, $primaryAssetData)
    {
        if (optional($secondaryAssetData->assetStatus)->slug != optional($primaryAssetData->assetStatus)->slug) {
            $response['status'] = 'error';
            $response['message'] = 'Merging is not possible for assets which are having different asset statuses. You have the option to purge a device if it is deemed to be a duplicated.';
            return $response;
        }

        // if (optional($primaryAssetData->assetStatus)->has_user == 1) {
        //     if ($secondaryAssetData->user_id != $primaryAssetData->user_id) {
        //         $response['status'] = 'error';
        //         $response['message'] = 'Merging is not possible for assets which are having different users. You have the option to purge a device if it is deemed to be a duplicated.';
        //         return $response;
        //     }
        // }

        // if (optional($primaryAssetData->assetStatus)->has_location == 1) {
        //     if ($secondaryAssetData->location_id != $primaryAssetData->location_id) {
        //         $response['status'] = 'error';
        //         $response['message'] = 'Merging is not possible for assets which are having different locations. You have the option to purge a device if it is deemed to be a duplicated.';
        //         return $response;
        //     }
        // }

        return [];
    }

    /**
     * create action buttons for assets
     * @param $item - asset for which buttons should be added
     * @param boolean $isParent
     */
    public function getActionButtons($item)
    {
        $actions = '';

        $actions = "<div style='width:155px'><a class='btn btn-link' data-id='" . $item->id . "' data-serial='" . $item->serial_no . "' data-tag='" . $item->asset_tag . "' data-toggle='modal' data-target='#editDuplicateAssetModal'><i class='icon icon-n-edit'></i></a>";

        if (!in_array($item->asset_status_slug, ['end_of_life_data_wiped', 'end_of_life'])) {
            $actions .= "<a class='btn btn-link btnDelete' data-id='" . $item->id . "' title='Purge' data-toggle='modal' data-target='#confirm-delete' data-url=" . route('purge-assets.destroy', $item->id) . "><i class='icon icon-delete-forever'></i></a>";
        }

        if ($item->is_parent) {
            $actions .= "<a class='btn btn-primary btn-sm' data-id='" . $item->id . "' data-child_ids='" . $item->matching_assets . "' data-toggle='modal' data-target='#mergeduplicateAssetModal'>Merge</a></div>";
        }


        return $actions;
    }

    /**
     * create action buttons for assets child assets
     * @param $item - asset for which buttons should be added
     * @param boolean $isParent
     */
    public function getActionButtonsForChild($asset)
    {
        $actions = '';

        $actions = "<a class='btn btn-link' data-id='" . $asset->id . "' data-serial='" . $asset->serial_no . "' data-tag='" . $asset->asset_tag . "' data-toggle='modal' data-target='#editDuplicateAssetModal'><i class='icon icon-n-edit'></i></a>";

        if (!in_array(optional($asset->assetStatus)->slug, ['end_of_life_data_wiped', 'end_of_life'])) {
            $actions .= "<a class='btn btn-link btnDelete' data-id='" . $asset->id . "' title='Purge' data-toggle='modal' data-target='#confirm-delete' data-url=" . route('purge-assets.destroy', $asset->id) . "><i class='icon icon-delete-forever'></i></a>";
        }
        return $actions;
    }

    /**
     * Updates the duplicated assets serial/tag
     * @param mixed $request
     * 
     * @return [type]
     */
    public function updateDuplicatedAssets($request)
    {
        try {
            $updateData = [];
            $asset = Asset::find(request('asset_id'));
            if ($asset->asset_tag != request('edit_duplicate_asset_asset_tag')) {
                $updateData['asset_tag'] = request('edit_duplicate_asset_asset_tag');
                $description = __('history.AssetTagUpdated', [
                    'assetId'     => $asset->id,
                    'oldAssetTag' => $asset->asset_tag,
                    'newAssetTag' => request('edit_duplicate_asset_asset_tag'),
                ]);
                $this->createAssetHistory('assetTag_updated', $asset, $asset->asset_tag, request('edit_duplicate_asset_asset_tag'), $description);
            }
            if ($asset->serial_no != request('edit_duplicate_asset_serial_no')) {
                $updateData['serial_no'] = request('edit_duplicate_asset_serial_no');
                $description = __('history.SerialNoUpdated', [
                    'assetId'     => $asset->id,
                    'assetTag'    => $asset->serial_no,
                    'oldSerialNo' => $asset->serial_no,
                    'newSerialNo' => request('edit_duplicate_asset_serial_no'),
                ]);
                $this->createAssetHistory('serialNo_updated', $asset, $asset->serial_no, request('edit_duplicate_asset_serial_no'), $description);
            }

            $asset->update($updateData);
            // $this->updateDuplicationsTable($asset);

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * creates the update history of duplicate asets serial/tag
     * @param mixed $action
     * @param mixed $asset
     * @param mixed $assetOldValue
     * @param mixed $assetNewValue
     * @param mixed $description
     * 
     * @return [type]
     */
    private function createAssetHistory($action, $asset, $assetOldValue, $assetNewValue, $description)
    {
        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'action' => $action,
            'old_value' => $assetOldValue,
            'new_value' => $assetNewValue,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
        return true;
    }
}
