<?php

namespace App\Services\Reports\DepreciationValue;

use App\Repositories\CommonFilterRepository;
use App\Repositories\Reports\DepreciationValueRepository;
use App\Services\ReportGraphAndDataExport;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;

class DepreciationValueChartService
{

    /**
     * Constructor for the class.
     *
     * @param DepreciationValueRepository $repository             The repository instance for handling depreciation values.
     * @param CommonFilterRepository      $commonFilterRepository The repository instance for handling common filters.
     */
    public function __construct(protected DepreciationValueRepository $repository, protected CommonFilterRepository $commonFilterRepository)
    {
    }

    /**
     * Retrieve data from the repository.
     *
     * @return array
     */
    public function data()
    {
        $assetTypes = $this->repository->getAssetTypes();
		// Commented Unused: $technical = null;.
		// Commented Unused: $formTitle = '';.
		// Commented Unused: $formId = '';.
        // Commented Unused: $requestData = $this->getRequestData(request()->all());.
        $techSpecs = $this->repository->getTechnicalSpecs();
        $hardWareStandards = $this->repository->getHardwareStandardsList();

        return compact('assetTypes', 'hardWareStandards', 'techSpecs');
    }

	/**
	 * Generate the function comment for the given function body.
	 *
	 * @return array The compacted data containing technical, graphArray, and yearsLabel.
	 */
	public function chartData()
	{
		$requestData = $this->getRequestData(request()->all());
		$technical = null;
		$graphArray = [];
		$dataArray = [];
		$yearsLabel = ["0", "< 0.5", "1", "2", "3", "4"];
		$formTitle = "";
		$formId = '';

		if ($requestData['asset_type']) {
			// Commented Unused: $hardWareStandards = $this->repository->getHardwareStandardsByAssetTypes($requestData['asset_type']);.
			$formTitle = $this->repository->findAssetTypeName($requestData['asset_type']);
			$formId = $requestData['asset_type'];
		}

		if ($requestData['manufacturer']) {
			$manufactures = $this->repository->filterManufactures($requestData['manufacturer']);
            $formId = implode('', $requestData['manufacturer']);

			if ($manufactures) {
				$formTitle = implode(', ', array_column($manufactures, 'name'));
			}
		}

		if (!empty($requestData['make_and_model'])) {
			$makeAndModels = $this->repository->filterMakeAndModels($requestData['make_and_model']);
            $formId = implode('', $requestData['make_and_model']);

			if ($makeAndModels) {
				$formTitle = implode(', ', array_column($makeAndModels, 'name'));
			}
		}

		if ($requestData['technical_spec']) {
			$technicalSpecs = $this->repository->filterTechnicalSpecs($requestData['technical_spec']);
			$title = [];

			foreach ($technicalSpecs as $tech) {
				$title[] = $tech->makeAndModel->makeModelName;
			}

			$formTitle = implode(', ', array_unique($title));
			$formId = $formTitle;
		}

		if ($formId == '') {
			return false;
		}

		$graphArray[$formTitle][0] = $formId;
		$graphArray[$formTitle][1] = [];

		$technical = $this->repository->getTechnicalSpecsQuery();
		$technical = $this->commonFilterRepository->filterWithDirectFields($technical, 'id', $requestData['technical_spec']);
		$technical = $this->commonFilterRepository->filterWithWhereHasRelationFields($technical, 'makeAndModel', 'id', $requestData['make_and_model']);
		$technical = $this->commonFilterRepository->filterWithWhereHasRelationFields($technical, 'makeAndModel', 'manufacturer_id', $requestData['manufacturer']);
		$technical = $this->commonFilterRepository->filterWithWhereHasRelationFields($technical, 'makeAndModel', 'asset_type_id', $requestData['asset_type']);
		$technical = $technical->get();

		if ($technical) {
			foreach ($technical as $tech) {
				$dataArray[] = $this->formattedData($tech);
			}

			$sum = [];

			if (count($dataArray)) {
				foreach ($dataArray as $item) {
					foreach ($item as $key => $value) {
						if (!isset($sum[$key])) {
							$sum[$key] = 0;
						}

						$sum[$key] += $value;
					}
				}

				foreach ($sum as $key => $item) {
					$formattedData[$key] = round($item / count($dataArray), 2);
				}

				$graphArray[$formTitle][0] = $formId;
				$graphArray[$formTitle][1] = $formattedData;
			}
		}

		return compact('technical', 'graphArray', 'yearsLabel');
	}

    /**
     * Retrieves and filters the request data.
     *
     * @param array $requestData The array containing the request data.
     *
     * @return array The filtered request data.
     */
    public function getRequestData(array $requestData): array
    {
        return [
            'asset_type' => $requestData['asset_type'] ?? '',
            'make_and_model' => array_filter((array) ($requestData['make_and_model'] ?? [])),
            'technical_spec' => array_filter((array) ($requestData['technical_spec'] ?? [])),
			'manufacturer' => array_filter((array) ($requestData['manufacturer'] ?? [])),
        ];
    }

    /**
     * Formats the data for a given technology.
     *
     * @param mixed $tech The technology to format the data for.
     *
     * @return array The formatted data.
     */
    public function formattedData($tech)
	{
		$years = [0, 0.5, 1, 2, 3, 4];
		$formattedData = [];

		$filteredData  = $this->repository->getAssetsForChart($tech);
		$assetAverageValue = ($filteredData->count != 0) ? $filteredData->total_original_value / $filteredData->count : 0;

		foreach ($years as $year) {
			if ($year > 0) {
				$depreciation = 0;

				if ($tech->life_of_asset != 0) {
					$depreciation = round((($assetAverageValue - ($assetAverageValue / 8)) / $tech->life_of_asset) * $year, 0);
				}

				$formattedData[] = $assetAverageValue > $depreciation ? $assetAverageValue - $depreciation : 0;
			} else {
				$formattedData[] = $assetAverageValue;
			}
		}

		return $formattedData;
	}

	/**
	 * Export the graph and data list to Excel.
	 *
	 * @param array $dataCollection The collection of data.
	 */
	public function exportGraphAndDataListToExcel(array $dataCollection)
	{
		$name = str_replace(' ', '_', request('name'));
		$name = str_replace('/', '_', $name);
		$sheetTitle = str_replace(' ', '_', request('sheettitle'));
		$sheetTitle = str_replace('/', '_', $sheetTitle);

		$headings = array_keys($dataCollection[0][0]);
		$image = storage_path('app/' . request('image'));
		$filename = request("filename") . time() . '.xlsx';
		$excel = Excel::download(new ReportGraphAndDataExport($dataCollection, $image, $headings, $sheetTitle), $filename, \Maatwebsite\Excel\Excel::XLSX);
		$img = explode('public/', request('image'));
		Storage::disk('public')->delete($img[1]);

		return $excel;
	}
}