<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Reports\ComputerDeployedRepository;

class ComputerDeployedService extends AbstractReportService
{

    protected $repository;
    protected $reportOutputData;
    protected $commonFilterRepository;

    /**
     * Constructs a new instance of the class.
     *
     * @param ComputerDeployedRepository $repository The deployment repository.
     * @param CommonFilterRepository $commonFilterRepository The commonFilterRepository instance
     * @param ReportOutputData $reportOutputData The reportOutputData instance
     */
    public function __construct(ComputerDeployedRepository $repository,  CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * filter and return assets data and totalcount
     *
     * @return array
     */
    public function data()
    {
        $result     = $this->repository->getAssetsData();
        $inputData  = $this->getInputData();
        $result     = $this->filterResults($result, $inputData);
        $count      = $result->count();
        $result     = $this->reportOutputData->getOutputData($result, ['assets.created_at' => 'desc']);

        return compact('result', 'count');
    }

    /**
     * Retrieves the input data for processing.
     *
     * @return array
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();

        $inputData =  [
            'asset_tag' => $requestedData['asset_tag'] ?? '',
            'serial_no'    => $requestedData['serial_no'] ?? '',
            'make_and_model_id' => $requestedData['make_and_model_id'] ?? '',
            'technical_spec_id'    => $requestedData['technical_spec_id'] ?? '',
            'user'     => $requestedData['user'] ?? '',
            'location_id'     => $requestedData['location_id'] ?? '',
            'work_location'  => $requestedData['work_location'] ?? '',
            'assigned_date_from'  => $requestedData['assigned_date_from'] ?? '',
            'assigned_date_to'  => $requestedData['assigned_date_to'] ?? '',
            'manufacturer'  => $requestedData['manufacturer'] ?? '',
        ];

        return $inputData;
    }

    /**
     * Filer the results based on filters
     *
     * @param $assets DB Query
     * @param Array $inputData   Filter data
     *
     * @return Query
     */
    public function filterResults($assets, $inputData)
    {
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_tag', $inputData['asset_tag']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'serial_no', $inputData['serial_no']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'make_and_model_id', $inputData['make_and_model_id']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'technical_spec_id', $inputData['technical_spec_id']);
        $assets = $this->commonFilterRepository->filterWithRelationalDateRange($assets, 'latestAssignedOrInstalledAssetHistory', 'created_at', $inputData['assigned_date_from'], $inputData['assigned_date_to']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', 'id', $inputData['user']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'user', 'city', $inputData['work_location']);
        $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'location_id', $inputData['location_id']);
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'makeAndModel', 'manufacturer_id', $inputData['manufacturer']);

        return $assets;
    }

    /**
     * Export the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $assets = $this->repository->getAssetsData();
        $inputData = $this->getInputData();
        $assets = $this->filterResults($assets, $inputData);
        $assets    = $assets->orderBy('assets.created_at', 'desc');

        return $assets;
    }

    /**
     * Format the record for export as csv
     * @param object single record object
     *
     * @return array formatted
     */
    public function getReportNestedData($asset)
    {
        $nestedData['report']['asset_tag'] = $nestedData['export']['Asset Tag #'] = disableCSVInjection($asset->serial_no);
        $nestedData['report']['serial_no'] = $nestedData['export']['Serial #'] = disableCSVInjection($asset->serial_no);
        $nestedData['report']['manufacturer'] = $nestedData['export']['Manufacturer'] = disableCSVInjection($asset->makeAndModel?->manufacturer?->name);
        $nestedData['report']['make_and_model'] = $nestedData['export']['Hardware Standard'] = disableCSVInjection(optional($asset->makeAndModel)->name ?? '');
        $nestedData['report']['technical_specs'] = $nestedData['export']['Technical Specs'] = disableCSVInjection(optional($asset->technicalSpec)->details ?? '');
        $nestedData['report']['user_location'] = $nestedData['export']['User/Location'] = disableCSVInjection($asset->user ? ($asset->user->first_name . ' ' . $asset->user->last_name) : ($asset->location ? $asset->location->room_name : ''));
        $nestedData['report']['work_location'] = $nestedData['export']['Work location'] = disableCSVInjection(optional($asset->user)->city ?? '');
        $nestedData['report']['date_assigned'] = $nestedData['export']['Date assigned'] = disableCSVInjection($asset->latestAssignedOrInstalledAssetHistory ? parse_date_from_db_datetime($asset->latestAssignedOrInstalledAssetHistory->getRawOriginal('created_at')) : '');

        return $nestedData;
    }



    /**
     * Create each row of output data
     * @params $assets,$index
     * @return array
     * */
    public function getNestedData($asset, $index)
    {
        $commonData = $this->getReportNestedData($asset);
        $nestedData = $commonData['report'];
        $nestedData['id'] = $index;
        $nestedData['asset_tag'] = generateAssetLink($asset->id, $nestedData['asset_tag']);
        $nestedData['serial_no'] = generateAssetLink($asset->id, $nestedData['serial_no']);
        $nestedData['user_location'] = ($asset->user) ? generateUserLink($asset->user?->id, $asset->user?->user_name) : generateLocationLink($asset->location?->id, $asset->location?->room_name);

        return $nestedData;
    }
}
