<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Models\Asset;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Reports\ComputerByStatusRepository;

/**
 * Service class for SHI Inventory report
 */
class ComputerByStatusService extends AbstractReportService
{
    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Constructs a new instance of the class.
     *
     * @param ComputerByStatusRepository $repository The repository instance.
     * @param CommonFilterRepository $commonFilterRepository The common filter repository instance.
     * @param ReportOutputData $reportOutputData The report output data instance.
     */
    public function __construct(ComputerByStatusRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Data to the Report data table
     *
     * @return void
     */
    public function data()
    {
        $requestedData  = request()->form;
        $inputData      = $this->getInputData($requestedData);
        $result         = $this->repository->getData();
        $result         = $this->filterWithInputData($result, $inputData);
        $count          = $result->count();
        $result         = $this->reportOutputData->getOutputData($result, ['id' => 'desc']);

        return compact('result', 'count');
    }

    /**
     * Filter the query results based on the filters applied
     *
     * @return object
     */
    public function filterWithInputData($query, array $inputData)
    {
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_type_id', $inputData['asset_type_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_status_id', $inputData['asset_status_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'make_and_model_id', $inputData['make_and_model_id']);
        $query = $this->repository->filterWithManufacturer($query, $inputData['manufacturer']);

        return $query;
    }

    /**
     * Taking the filter field data for listing
     */
    public function getInputData(array $requestData): array
    {
        return [
            'asset_type_id'     => $requestData['asset_type'] ?? '',
            'asset_status_id'   => $requestData['status'] ?? '',
            'make_and_model_id' => $requestData['make_and_model'] ?? '',
            'manufacturer'      => $requestData['manufacturer'] ?? '',
        ];
    }

    /**
     * Retrieves the nested data for generating a report.
     *
     * @param mixed $asset The asset object.
     * @return array The nested data for the report.
     */
    public function getReportNestedData($asset): array
    {
        $nestedData = [];
        $nestedData['report']['type']               = $nestedData['export']['Type'] = $asset?->assetType?->name;
        $nestedData['report']['po_number']          = $nestedData['export']['PO #'] = $asset->po_id;
        $nestedData['report']['asset_tag']          = $nestedData['export']['Asset Tag #'] = $asset->asset_tag;
        $nestedData['report']['child_asset']        = $nestedData['export']['Child Asset #'] = $asset->child;
        $nestedData['report']['serial_number']      = $nestedData['export']['Serial #'] = $asset->serial_no;
        $nestedData['report']['warranty_end_date']  = $nestedData['export']['Warranty End Date'] = $asset->warranty_end_date;
        $nestedData['report']['hardware_standard']  = $nestedData['export']['Hardware Standard'] = $asset->makeAndModel?->makeModelName;
        $nestedData['report']['technical_spec']     = $nestedData['export']['Technical Spec'] = $asset->technicalSpec?->details;
        $nestedData['report']['asset_status']       = $nestedData['export']['Asset Status'] = $asset?->assetStatus?->name;
        $nestedData['report']['user_location']      = $nestedData['export']['User/Location'] = $asset->user ? ($asset->user->first_name . ' ' . $asset->user->last_name) : ($asset->location ? $asset->location->room_name : '');
        $nestedData['report']['age']                = $nestedData['export']['Age'] = $asset->AssetAge;
        $nestedData['report']['modified_date']      = $nestedData['export']['Last Modified Date'] = $asset?->latestAssetHistory?->updated_at;
        $nestedData['report']['modified_by']        = $nestedData['export']['Last Modified By'] = ($asset?->latestAssetHistory?->user?->first_name . ' ' . $asset?->latestAssetHistory?->user?->last_name) . (($asset->location) ? ' / ' . $asset->location->room_name : '');

        return $nestedData;
    }

    /**
     * Retrieves the nested data for a given item.
     *
     * @param mixed $item The item for which to retrieve the nested data.
     * @param int $index The index of the item.
     * @return array The nested data for the item.
     */
    public function getNestedData($item, $index)
    {
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['serial_number']    = '<a href="' . route('assets.show', $item->id) . '">' . $item->serial_no . '</a>';
        $nestedData['asset_tag']        = '<a href="' . route('assets.show', $item->id) . '">' . $item->asset_tag . '</a>';
        $nestedData['id'] = $index;

        return $nestedData;
    }

    /**
     * Retrieves the output data based on the provided assets and data.
     *
     * @param mixed[] $assets The array of assets.
     * @param array $data The array of data.
     * @return mixed[] The modified array of data.
     */
    public function getOutputData($assets, array $data)
    {
        $parentIndex = 0;

        foreach ($assets as $asset) {
            ++$parentIndex;
            $nestedData = $this->getNestedData($asset, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Export the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $requestedData = request()->all();
        $inputData = $this->getInputData($requestedData);
        $assets = $this->repository->getData();
        $assets = $this->filterWithInputData($assets, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }
}
