<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Services\Reports\AbstractReportService;
use App\Repositories\Reports\ComputerByAgeRepository;

class ComputerByAgeService extends AbstractReportService
{
    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Constructs a new instance of the class.
     *
     * @param ComputerByAgeRepository $repository The computer by age repository.
     * @param CommonFilterRepository $commonFilterRepository The common filter repository.
     * @param ReportOutputData $reportOutputData The report output data.
     */
    public function __construct(ComputerByAgeRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Retrieves the requested data and filters it based on the input data.
     *
     * @return array Returns an array containing the filtered assets and the count of assets.
     */
    public function data()
    {
        $requestedData  = request()->form;
        $inputData      = $this->getInputData($requestedData);
        $result         = $this->repository->getData();
        $result         = $this->filterWithInputData($result, $inputData);
        $count          = $result->count();
        $result         = $this->reportOutputData->getOutputData($result, ['id' => 'desc']);

        return compact('result', 'count');
    }

    /**
     * Filter the query results based on the filters applied
     *
     * @param  mixed $assets
     * @param  mixed $inputData
     * @return object
     */
    public function filterWithInputData($query, $inputData)
    {
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'serial_no', $inputData['serial_no']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_status_id', $inputData['status']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_tag', $inputData['asset_tag']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'user_id', $inputData['user_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'location_id', $inputData['location_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'make_and_model_id', $inputData['make_and_model_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'technical_spec_id', $inputData['technical_spec_id']);
        $query = $this->commonFilterRepository->filterWithWhereHasRelationFields($query, 'user', 'city', $inputData['work_location']);
        $query = $this->commonFilterRepository->filterAssetsByAgeGroup($query, 'created_at',$inputData['asset_age']);

        return $query;
    }

    /**
     * Retrieves the input data from the requested data array.
     *
     * @param array $requestedData The requested data array.
     * @return array The input data array containing the following keys:
     */
    public function getInputData(array $requestedData): array
    {
        return [
            'status'            => $requestedData['status'] ?? '',
            'serial_no'         => $requestedData['serial_no'] ?? '',
            'asset_tag'         => $requestedData['asset_tag'] ?? '',
            'user_id'           => $requestedData['user'] ?? '',
            'location_id'       => $requestedData['location'] ?? '',
            'make_and_model_id' => $requestedData['make_and_model'] ?? '',
            'technical_spec_id' => $requestedData['technical_spec'] ?? '',
            'asset_age'         => $requestedData['asset_age'] ?? '',
            'work_location'     => $requestedData['work_location'] ?? '',
        ];
    }

    /**
     * Retrieves the nested data for generating a report.
     *
     * @param $asset The asset object for which the report is being generated.
     */
    public function getReportNestedData($asset)
    {
        $nestedData = [];
        $nestedData['report']['asset_tag']          = $nestedData['export']['Asset Tag #'] = $asset->asset_tag;
        $nestedData['report']['serial_no']          = $nestedData['export']['Serial #'] = $asset->serial_no;
        $nestedData['report']['hardware_standard']  = $nestedData['export']['Hardware Standard'] = $asset->makeAndModel?->makeModelName;
        $nestedData['report']['tech_specs']         = $nestedData['export']['Technical Specs'] = $asset->technicalSpec?->details;
        $nestedData['report']['user_location']      = $nestedData['export']['User/Location'] = $asset->user ? ($asset->user->first_name . ' ' . $asset->user->last_name) : ($asset->location ? $asset->location->room_name : '');
        $nestedData['report']['user_work_location'] = $nestedData['export']['User Work Location'] = $asset->user?->city;
        $nestedData['report']['asset_status']       = $nestedData['export']['Asset Status'] = $asset->assetStatus?->name;
        $nestedData['report']['created_date']       = $nestedData['export']['Date Created'] = $asset->created_at;
        $nestedData['report']['age']                = $nestedData['export']['Asset Age'] = $asset->AssetAge;

        return $nestedData;
    }

    /**
     * Retrieves the nested data for a given item.
     *
     * @param mixed $item The item for which to retrieve the nested data.
     * @param int $index The index of the item.
     * @return array The nested data for the item.
     */
    public function getNestedData($item, $index)
    {
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['serial_no'] = generateUserLink($item->id, $nestedData['serial_no']);
        $nestedData['asset_tag'] = generateUserLink($item->id, $nestedData['asset_tag']);
        $nestedData['id'] = $index;
        $nestedData['user_location'] = $item->user ? generateUserLink($item->user?->id, $item->user?->user_name) : generateLocationLink($item->location?->id, $item->location?->room_name);

        return $nestedData;
    }

    /**
     * Create output data
     * @params $assets,$start, $data, $type
     * @return array
     * */
    public function getOutputData($assets, $data)
    {
        $parentIndex = 0;

        foreach ($assets as $asset) {
            ++$parentIndex;
            $nestedData = $this->getNestedData($asset, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Export the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $requestedData = request()->all();
        $inputData = $this->getInputData($requestedData);
        $assets = $this->repository->getData();
        $assets = $this->filterWithInputData($assets, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }
}
